import traceback
from collections import OrderedDict
from urllib.parse import urlencode

import flask
import requests


def set_up(app, group_auth=True):
    session = requests.Session()
    adapter = requests.adapters.HTTPAdapter(
        max_retries=requests.packages.urllib3.Retry(
            **app.config['BLACKBOX_RETRY_PARAMS']
        )
    )
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    app.blackbox_requests_session = session
    app.before_request(lambda: check_auth(group_auth))


def check_auth(group_auth):
    """
    Check current request is authorized using blackbox api.

    Sets ``flask.g.username`` to current login username as a sideeffect.

    :returns: None in case user is authorized. Otherwise returns redirect
        response object to a passport page.
    """
    if flask.current_app.config.get('BYPASS_AUTH'):
        flask.g.username = flask.current_app.config['BYPASS_AUTH_AS']
        flask.g.usergroups = flask.current_app.config.get(
            'BYPASS_AUTH_AS_GROUPS', []
        )
        return None

    blackbox_args = None
    blackbox_headers = {}

    Session_id = flask.request.cookies.get('Session_id', None)
    sessionid2 = flask.request.cookies.get('sessionid2', None)
    if Session_id:
        # cookie-based authorization
        host = flask.request.environ.get('HTTP_HOST')
        if not host:
            host = flask.current_app.config['SERVER_NAME']
        host = host.split(':')[0]
        # http://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml
        blackbox_args = OrderedDict((
            ('method', 'sessionid'),
            ('sessionid', Session_id),
            ('userip', flask.request.access_route[-1]),
            ('host', host),
            ('format', 'json'),
        ))
        if sessionid2:
            blackbox_args['sslsessionid'] = sessionid2
    elif 'Authorization' in flask.request.headers:
        # OAuth authorization
        # https://doc.yandex-team.ru/blackbox/reference/method-oauth.xml
        blackbox_args = OrderedDict((
            ('method', 'oauth'),
            ('userip', flask.request.access_route[-1]),
            ('format', 'json'),
        ))
        blackbox_headers['Authorization'] = flask.request.headers['Authorization']

    if blackbox_args is not None:
        blackbox_uri = '{}?{}'.format(flask.current_app.config['BLACKBOX_URI'],
                                      urlencode(blackbox_args))
        try:
            result = flask.current_app.blackbox_requests_session.get(
                blackbox_uri,
                timeout=flask.current_app.config['BLACKBOX_TIMEOUT'],
                verify=True,
                headers=blackbox_headers
            ).json()
        except Exception as exc:
            flask.g.username = None
            flask.g.usergroups = []
            flask.g.auth_error = {
                'exc_class': type(exc).__name__,
                'exc_message': str(exc),
                'traceback': traceback.format_exc(),
            }
            return None
        else:
            if result['error'] == 'OK':
                if result['status']['value'] == 'VALID':
                    flask.g.username = result['login']
                    if group_auth:
                        try:
                            flask.g.usergroups = get_groups(
                                flask.g.username, flask.current_app.config
                            )
                        except Exception as exc:
                            flask.g.usergroups = []
                            flask.g.group_auth_error = {
                                'exc_class': type(exc).__name__,
                                'exc_message': str(exc),
                                'traceback': traceback.format_exc(),
                            }
                    return None
                if result['status']['value'] == 'NEED_RESET':
                    url = flask.current_app.config['RENEW_AUTH_COOKIE_URI']
                    args = urlencode({'retpath': flask.request.url})
                    return flask.redirect('{}?{}'.format(url, args))

    passport_url = flask.current_app.config['PASSPORT_URI']
    args = urlencode({'retpath': flask.request.url})
    return flask.redirect('{}?{}'.format(passport_url, args))


def get_groups(username, cfg):
    # https://staff-api.yandex-team.ru/v3/persons?_doc=1
    params = {'login': username,
              '_one': '1',
              '_fields': 'groups.group.url'}
    resp = requests.get(cfg['STAFF_URI'] + '/persons',
                        params=params,
                        headers=cfg['STAFF_HEADERS'],
                        timeout=cfg['STAFF_TIMEOUT'],
                        allow_redirects=False)
    resp.raise_for_status()
    return sorted(filter(
        None,
        (rec['group'].get('url') for rec in resp.json()['groups'])
    ))
