import difflib
import re
import time
from datetime import datetime, timedelta

import jinja2
import babel.dates
import markdown
import yaml
import flask
from flask.ext.babel import get_locale, get_timezone


def format_timedelta(timestamp):
    td = timedelta(seconds=timestamp - time.time())
    return babel.dates.format_timedelta(td, add_direction=True,
                                        locale=get_locale())


def format_timestamp(timestamp, format='medium'):
    dt = datetime.utcfromtimestamp(timestamp)
    return babel.dates.format_datetime(dt, format=format, locale=get_locale(),
                                       tzinfo=get_timezone())


_words = re.compile(r'[^\s]+[^\n]|\n')
_lines = re.compile(r'.*\n?')
_lineends = re.compile('\n|\r\n|\r')

TEXTDIFF_UNIQUE_FMT = '<span class="unique">{}</span>'
TEXTDIFF_CHANGED_FMT = '<span class="changed">{}</span>'
TEXTDIFF_EQUAL_FMT = '{}'

def format_text_diff(a, b):
    a = jinja2.escape(_lineends.sub('\n', a))
    b = jinja2.escape(_lineends.sub('\n', b))
    a = [line.strip('\n') for line in _lines.findall(a) if line]
    b = [line.strip('\n') for line in _lines.findall(b) if line]
    sm = difflib.SequenceMatcher(None, a, b)
    res = []
    for tag, i1, i2, j1, j2 in sm.get_opcodes():
        if tag == 'delete':
            res.append((TEXTDIFF_UNIQUE_FMT, '\n'.join(a[i1:i2])))
        elif tag == 'equal':
            res.append((TEXTDIFF_EQUAL_FMT, '\n'.join(a[i1:i2])))
        elif tag == 'replace':
            awords = _words.findall('\n'.join(a[i1:i2]))
            bwords = _words.findall('\n'.join(b[j1:j2]))
            wres = []
            smwords = difflib.SequenceMatcher(None, awords, bwords)
            for tag, i1, i2, j1, j2 in smwords.get_opcodes():
                if tag == 'delete':
                    wres.append((TEXTDIFF_UNIQUE_FMT, awords[i1:i2]))
                elif tag == 'equal':
                    wres.append((TEXTDIFF_EQUAL_FMT, awords[i1:i2]))
                elif tag == 'replace':
                    achars = ''.join(awords[i1:i2])
                    bchars = ''.join(bwords[j1:j2])
                    smchars = difflib.SequenceMatcher(None, achars, bchars)
                    if smchars.ratio() < 0.75:
                        wres.append((TEXTDIFF_CHANGED_FMT, awords[i1:i2]))
                        continue
                    for tag, i1, i2, j1, j2 in smchars.get_opcodes():
                        if tag == 'delete':
                            wres.append((TEXTDIFF_UNIQUE_FMT, achars[i1:i2]))
                        elif tag == 'equal':
                            wres.append((TEXTDIFF_EQUAL_FMT, achars[i1:i2]))
                        elif tag == 'replace':
                            wres.append((TEXTDIFF_CHANGED_FMT, achars[i1:i2]))
            wres = ''.join(fmt.format(''.join(bits)) for fmt, bits in wres)
            res.append((TEXTDIFF_EQUAL_FMT, wres))
    return jinja2.Markup(''.join(fmt.format(bit + '\n') for fmt, bit in res))


def markdown_safe(text):
    md = markdown.Markdown(output_format='html5')
    return jinja2.Markup(md.convert(jinja2.escape(text)))


def yaml_dump(data, default_flow_style=True):
    return yaml.dump(data, default_flow_style=default_flow_style)


def in_the_past(timestamp):
    return time.time() > timestamp


def current_user_is_in_list(userlist):
    return flask.current_app.storage.is_user_in_userlist(
        flask.g.username, flask.g.usergroups, userlist
    )


jinja_filters = {
    'timedeltaformat': format_timedelta,
    'timestampformat': format_timestamp,
    'diffformat': format_text_diff,
    'markdown_safe': markdown_safe,
    'yaml_dump': yaml_dump,
}

jinja_tests = {
    'in_the_past': in_the_past,
}

jinja_globals = {
    'current_user_is_in_list': current_user_is_in_list,
}
