import flask
import requests
import wtforms
import jinja2
import yaml
from wtforms import validators
from flask_wtf import Form
from flask.ext.babel import gettext, lazy_gettext as _

from genisys.web import sandbox


class _SpaceSeparatedListField(wtforms.Field):
    def _value(self):
        if self.data:
            return ' '.join(str(item) for item in self.data)
        else:
            return ''

    def process_formdata(self, valuelist):
        if valuelist and valuelist[0]:
            self.data = valuelist[0].split()
            self.data = self._validate_data(self.data)
        else:
            self.data = []

    def _validate_data(self, data):
        return data


class UserListField(_SpaceSeparatedListField):
    widget = wtforms.widgets.TextInput()

    def __init__(self, name, **kwargs):
        kwargs.setdefault('description',
                          gettext('Usernames, separated by space. Groups in '
                                  'a form "group:url", like group:svc_skynet'))
        super(UserListField, self).__init__(name, **kwargs)

    def _validate_data(self, usernames):
        items = set(usernames)
        cfg = flask.current_app.config
        if cfg.get('BYPASS_STAFF_USERNAMES_CHECK'):
            return sorted(items)
        groupnames = set()
        usernames = set()
        for name in items:
            if "'" in name or '"' in name or ',' in name:
                raise wtforms.ValidationError(gettext("Invalid characters"))
            if name.startswith('group:'):
                name = name[len('group:'):]
                groupnames.add(name)
            else:
                usernames.add(name)

        err_msgs = []
        items = []
        if usernames:
            # https://staff-api.yandex-team.ru/v3/persons?_doc=1
            params = {'login': ','.join(sorted(usernames)),
                      '_fields': 'login',
                      'official.is_dismissed': 'false',
                      '_limit': len(usernames)}
            resp = requests.get(cfg['STAFF_URI'] + '/persons',
                                params=params,
                                headers=cfg['STAFF_HEADERS'],
                                timeout=cfg['STAFF_TIMEOUT'],
                                allow_redirects=False)
            if not resp.status_code == 200:
                err_msgs.append(gettext('Could not validate usernames'))
            else:
                existing_users = set(rec['login']
                                     for rec in resp.json()['result'])
                missing_users = usernames - existing_users
                if missing_users:
                    missing = ', '.join(sorted(missing_users))
                    if len(missing_users) == 1:
                        err_msgs.append(
                            gettext('Invalid username: %(username)s',
                                    username=missing)
                        )
                    else:
                        err_msgs.append(
                            gettext('Invalid usernames: %(usernames)s',
                                    usernames=missing)
                        )
                items.extend(sorted(existing_users))

        num_actual_users = len(items)

        if groupnames:
            # https://staff-api.yandex-team.ru/v3/groups?_doc=1
            params = {'_fields': 'url,affiliation_counters',
                      'is_deleted': 'false',
                      'url': ",".join(name for name in sorted(groupnames)),
                      '_limit': len(groupnames)}
            resp = requests.get(cfg['STAFF_URI'] + '/groups',
                                params=params,
                                headers=cfg['STAFF_HEADERS'],
                                timeout=cfg['STAFF_TIMEOUT'],
                                allow_redirects=False)
            if not resp.status_code == 200:
                raise wtforms.ValidationError(
                    gettext('Could not validate group names')
                )
            else:
                resp = resp.json()['result']
                existing_groups = set(rec['url'] for rec in resp)
                missing_groups = groupnames - existing_groups
                if missing_groups:
                    missing = ', '.join(sorted(missing_groups))
                    if len(missing_groups) == 1:
                        err_msgs.append(
                            gettext('Invalid group name: %(groupname)s',
                                    groupname=missing)
                        )
                    else:
                        err_msgs.append(
                            gettext('Invalid group names: %(groupnames)s',
                                    groupnames=missing)
                        )
                too_broad_groups = []
                for rec in resp:
                    users_in_group = sum(rec['affiliation_counters'].values())
                    num_actual_users += users_in_group
                    if users_in_group > cfg['EMAIL_MAX_RECIPIENTS']:
                        too_broad_groups.append(rec['url'])
                if len(too_broad_groups) == 1:
                    err_msgs.append(gettext('Too broadly defined group: '
                                            '%(groupname)s',
                                            groupname=too_broad_groups[0]))
                elif len(too_broad_groups) > 1:
                    groupnames = ', '.join(sorted(too_broad_groups))
                    err_msgs.append(gettext('Too broadly defined groups: '
                                            '%(groupnames)s',
                                            groupnames=groupnames))
                items.extend('group:' + name
                             for name in sorted(existing_groups))

        if num_actual_users > cfg['EMAIL_MAX_RECIPIENTS']:
            err_msgs.append(gettext(
                'Too many users in a list (%(actual)s > %(maximum)s)',
                actual=num_actual_users, maximum=cfg['EMAIL_MAX_RECIPIENTS']
            ))

        if err_msgs:
            raise wtforms.ValidationError('. '.join(err_msgs))

        return items


def brackets(form, field):
    opening = set('{[(')
    closing = {'}': '{', ']': '[', ')': '('}
    stack = []
    for c in field.data:
        if c in opening:
            stack.append(c)
            continue
        matching_open = closing.get(c)
        if matching_open is None:
            continue
        if not stack or stack[-1] != matching_open:
            stack = True
            break
        stack.pop()
    if stack:
        raise validators.ValidationError('Brackets syntax is broken')


class BlinovCalcSelectorField(wtforms.TextAreaField):
    # allow only printable ascii characters, check brackets syntax
    validators = [
        validators.Regexp(r'^[ -~\t\r\n]*$',
                          message=_("Only ASCII symbols are allowed")),
        brackets
    ]


class YamlTextArea(wtforms.widgets.TextArea):
    def __call__(self, field, **kwargs):
        kwargs['class_'] = kwargs.get('class_', '') + ' yaml'
        return super(YamlTextArea, self).__call__(field, **kwargs)


class YamlField(wtforms.TextAreaField):
    widget = YamlTextArea()

    def _value(self):
        if self.data:
            return self.data['source']
        else:
            return ''

    def process_formdata(self, valuelist):
        if not valuelist or not valuelist[0]:
            self.data = {'source': '', 'dict': {}}
            return

        self.data = {'source': valuelist[0], 'dict': None}
        try:
            self.data['dict'] = yaml.safe_load(self.data['source'])
        except yaml.error.MarkedYAMLError as exc:
            raise wtforms.ValidationError(
                gettext('Line %(linenum)d: %(problem)s',
                        linenum=exc.problem_mark.line,
                        problem=exc.problem)
            )
        except yaml.error.YAMLError as exc:
            raise wtforms.ValidationError(
                gettext('Yaml parser error: %(exc)s', exc=exc)
            )
        except Exception as exc:
            raise wtforms.ValidationError(
                gettext('Yaml parser exception: %(exc)s', exc=exc)
            )


class OldRevisionField(wtforms.IntegerField):
    widget = wtforms.widgets.HiddenInput()

    def __init__(self, **kwargs):
        vs = [validators.required()]
        super(OldRevisionField, self).__init__(validators=vs, **kwargs)


class RuleNumbersListField(_SpaceSeparatedListField):
    widget = wtforms.widgets.HiddenInput()

    def _validate_data(self, data):
        if not all(item.isdigit() for item in data):
            raise wtforms.ValidationError(gettext('invalid integers in list'))
        result = [int(item) for item in data]
        return result


class SandboxResourceTypeField(wtforms.StringField):
    def _value(self):
        return self.data or ''

    def process_formdata(self, valuelist):
        if not valuelist or not valuelist[0]:
            self.data = None
            return
        self.data = valuelist[0]
        try:
            sandbox.get_resource_type_description(self.data)
        except sandbox.SandboxError as exc:
            raise wtforms.ValidationError(str(exc))


class SandboxReleasedResourceField(wtforms.SelectField):
    pass


class SandboxResourceField(wtforms.IntegerField):
    def process_formdata(self, valuelist):
        self.data = None
        if not valuelist or not valuelist[0]:
            return
        super(SandboxResourceField, self).process_formdata(valuelist)
        resource_id = self.data
        try:
            info = sandbox.get_resource_info(resource_id)
        except sandbox.SandboxError as exc:
            raise wtforms.ValidationError(str(exc))
        if info['type'] != self.resource_type:
            raise wtforms.ValidationError(gettext(
                'Specified resource is of type %(actual)s, not %(expected)s',
                actual=info['type'], expected=self.resource_type
            ))
        self.resource_id = resource_id
        self.resource_description = info['description']


class DescriptionForm(Form):
    action = wtforms.HiddenField()
    revision = OldRevisionField()
    desc = wtforms.TextAreaField(
        _('Description'), description=_('Markdown is supported')
    )


class OwnersForm(Form):
    action = wtforms.HiddenField()
    revision = OldRevisionField()
    owners = UserListField(_('Owners'))


class NewSubsectionForm(Form):
    STYPE_YAML = 'yaml'
    STYPE_SANDBOX_RESOURCE = 'sandbox_resource'

    action = wtforms.HiddenField()
    parent_revision = OldRevisionField()

    name = wtforms.StringField(
        _('Subsection name'), [validators.required()],
        description=_('Must be unique within a parent '
                      'section. Must not contain dot')
    )
    stype = wtforms.RadioField(
        _("Configuration type"), [validators.required()],
        choices=[(STYPE_YAML, _("Yaml")),
                 (STYPE_SANDBOX_RESOURCE, _("Sandbox resource"))],
        default="yaml",
    )
    sandbox_resource_type = SandboxResourceTypeField(
        _("Sandbox resource type")
    )
    owners = UserListField(_('Owners'))
    desc = wtforms.TextAreaField(
        _('Description'), description=_('Markdown is supported')
    )

    def validate_sandbox_resource_type(self, field):
        if self['stype'].data == 'sandbox_resource':
            if self['sandbox_resource_type'].data is None:
                raise wtforms.ValidationError(
                    gettext('This field is required '
                            'for "Sandbox resource" configuration type')
                )
        else:
            if self['sandbox_resource_type'].data is not None:
                raise wtforms.ValidationError(
                    gettext('This field is irrelevant for configuration type '
                            'other than "Sandbox resource"')
                )


class _BaseRuleConfigForm(Form):
    revision = OldRevisionField()
    action = wtforms.HiddenField()


# ugly hack to make this field appear in BaseNewRuleForm above fields,
# inherited from parent class EditRuleForm
_rule_name_field = wtforms.StringField(_('Name'), [validators.required()])


class EditRuleForm(Form):
    HTYPE_ALL = 'all'
    HTYPE_SOME = 'some'

    revision = OldRevisionField()
    action = wtforms.HiddenField()
    desc = wtforms.TextAreaField(
        _('Description'), description=_('Markdown is supported')
    )
    editors = UserListField(_('Editors'))
    htype = wtforms.RadioField(
        _("Host list"), [validators.required()],
        choices=[(HTYPE_ALL, _("Apply to all hosts")),
                 (HTYPE_SOME, _("Specify selector of hosts to apply to"))],
        default=HTYPE_SOME,
    )
    selector = BlinovCalcSelectorField(_('Selector'))

    def validate_selector(self, field):
        if self['htype'].data == self.HTYPE_SOME:
            if not field.data:
                raise wtforms.ValidationError(
                    gettext('This field is required for chosen host list type')
                )


class _BaseNewRuleForm(EditRuleForm):
    name = _rule_name_field
del _rule_name_field


class _BaseYamlRuleForm(Form):
    config = YamlField(_('Config'), [validators.required()],
                       description=_('In yaml format'))

    def __init__(self, config_source=None, data=None, **kwargs):
        if config_source:
            if data is None:
                data = {}
            data.update({'config': {'source': config_source}})
        kwargs['data'] = data
        super(_BaseYamlRuleForm, self).__init__(**kwargs)

    def get_config(self):
        return self.config.data['dict']

    def get_config_source(self):
        return self.config.data['source']


class NewYamlRuleForm(_BaseYamlRuleForm, _BaseNewRuleForm):
    pass


class YamlRuleConfigForm(_BaseYamlRuleForm, _BaseRuleConfigForm):
    pass


class _BaseSandboxResourceRuleForm(Form):
    class Meta(wtforms.meta.DefaultMeta):
        def bind_field(self, form, unbound_field, options):
            sup = super(_BaseSandboxResourceRuleForm.Meta, self)
            bound_field = sup.bind_field(form, unbound_field, options)
            if isinstance(bound_field, SandboxReleasedResourceField):
                bound_field.choices = [('', '')] + [
                    (str(r['resource_id']),
                     gettext("#%(resource_id)s (%(description)s)", **r))
                    for r in form.released_resources
                ]
            elif isinstance(bound_field, SandboxResourceField):
                bound_field.resource_type = form.resource_type
            elif bound_field.name == 'alias':
                bound_field.choices = [('', '')] + [
                    (alias['id'], alias['name']) for alias in form.aliases
                ]
            return bound_field

    rtype = wtforms.RadioField(
        _("Sandbox resource"),
        choices=[
            ("released", _("Choose from resources released lately")),
            ("by_id", _("Specify resource id")),
            ("by_alias", _("By released resource alias")),
        ],
        default="released"
    )
    resource = SandboxResourceField(_("Resource id"))
    released_resource = SandboxReleasedResourceField(_("Released resource"))
    alias = wtforms.SelectField(_("Alias"))

    _auto_formdata = object()

    def __init__(self, resource_type, released_resources, aliases,
                 config_source=None, data=None,
                 formdata=_auto_formdata, **kwargs):
        self.resource_type = resource_type
        self.released_resources = released_resources
        self.aliases = aliases
        self.aliases_by_id = {alias['id']: alias for alias in aliases}
        if formdata is self._auto_formdata:
            if flask.request and flask.request.method == 'POST':
                formdata = flask.request.form.copy()
        if formdata and formdata is not self._auto_formdata:
            if formdata.get('rtype') == 'released':
                formdata['resource'] = ''
                formdata['alias'] = ''
            elif formdata.get('rtype') == 'by_alias':
                formdata['resource'] = ''
                formdata['released_resource'] = ''
            elif formdata.get('rtype') == 'by_id':
                formdata['released_resource'] = ''
                formdata['alias'] = ''
            kwargs['formdata'] = formdata
        elif config_source:
            if data is None:
                data = {}
            data.update({
                'rtype': config_source['rtype'],
                'released_resource': config_source.get('resource', ''),
                'resource': config_source.get('resource'),
                'alias': config_source.get('alias_id'),
            })
        kwargs['data'] = data
        super(_BaseSandboxResourceRuleForm, self).__init__(**kwargs)
        if not self.released_resources:
            del self.released_resource
            self.rtype.choices = [(key, value)
                                  for (key, value) in self.rtype.choices
                                  if key != 'released']
            self.rtype.default = 'by_id'
            self.rtype.description = jinja2.Markup(gettext(
                "List of lately released resources of  type %(rtype)s is not "
                "available yet. Come back in a couple of minutes, or meditate "
                'on the <a href="%(url)s">status page</a>, if you want '
                'to choose from a list of released resources. Otherwise, '
                'enter resource id in the field below.',
                rtype=jinja2.escape(resource_type),
                url=jinja2.escape(
                    flask.url_for("volatile_status", vtype='sandbox_releases',
                                  key=resource_type)
                )
            ))
        if not self.aliases:
            del self.alias
            self.rtype.choices = [(key, value)
                                  for (key, value) in self.rtype.choices
                                  if key != 'by_alias']

    def get_config(self):
        if self.data['rtype'] == 'released':
            return {'resource_id': int(self.released_resource.data)}
        elif self.data['rtype'] == 'by_alias':
            rid = self.aliases_by_id[self.alias.data]['resource_id']
            return {'resource_id': rid}
        else:
            return {'resource_id': self.resource.resource_id}

    def get_config_source(self):
        config_source = {'rtype': self.data['rtype']}
        if self.data['rtype'] == 'released':
            config_source['resource'] = int(self.released_resource.data)
            rrdict = {rr['resource_id']: rr['description']
                      for rr in self.released_resources}
            config_source['description'] = rrdict[config_source['resource']]
        elif self.data['rtype'] == 'by_alias':
            alias = self.aliases_by_id[self.data['alias']]
            config_source['alias_id'] = alias['id']
            config_source['alias_name'] = alias['name']
            config_source['resource'] = alias['resource_id']
            config_source['description'] = alias['resource_description']
        else:
            config_source['resource'] = self.resource.resource_id
            config_source['description'] = self.resource.resource_description
        return config_source

    def validate_resource(self, field):
        if self['rtype'].data == 'by_id':
            if field.data is None and not field.errors:
                raise wtforms.ValidationError(_('This field is required'))

    def validate_released_resource(self, field):
        if self['rtype'].data == 'released':
            if field.data == '' and not field.errors:
                raise wtforms.ValidationError(_('This field is required'))

    def validate_alias(self, field):
        if self['rtype'].data == 'by_alias':
            if field.data == '' and not field.errors:
                raise wtforms.ValidationError(_('This field is required'))


class NewSandboxResourceRuleForm(_BaseSandboxResourceRuleForm,
                                 _BaseNewRuleForm):
    pass


class SandboxResourceRuleConfigForm(_BaseSandboxResourceRuleForm,
                                    _BaseRuleConfigForm):
    pass


class AliasForm(Form):
    class Meta(_BaseSandboxResourceRuleForm.Meta):
        pass

    id = wtforms.HiddenField()
    name = wtforms.StringField(_("Alias name"), [validators.required()])
    resource = SandboxReleasedResourceField(_("Released resource"))

    def __init__(self, released_resources, *args, **kwargs):
        self.released_resources = released_resources
        super(AliasForm, self).__init__(*args, **kwargs)

    def validate_resource(self, field):
        if not field.data and not field.errors:
            raise wtforms.ValidationError(_('This field is required'))


class RuleOrderForm(Form):
    revision = OldRevisionField()
    order = RuleNumbersListField()
    action = wtforms.HiddenField()


class DeleteRuleForm(Form):
    revision = OldRevisionField()
    action = wtforms.HiddenField()
