# coding: utf-8
import collections
import contextlib
import uuid
import traceback
import hashlib
import codecs
import re
import copy
import time
import itertools

import pymongo
import pymongo.errors
import msgpack

from genisys.web import errors


@contextlib.contextmanager
def audit(database, who, who_groups, when, what, path, revision,
          affected_rules=(), extra=None):
    """
    Context manager for saving audit information for data modification queries.

    :param who: username of a person who makes the change.
    :param who: list of groups a person making the change belongs to
        at a moment of an action.
    :param when: modification timestamp. In general it should be the same as
        one saved in the document being modified.
    :param what: description of the action user is trying to perform,
        like "create_section".
    :param path: path of the section being modified.
    :param revision: revision number of the section being modified (before
        change was made).
    :param affected_rules: list of rule names affected by this change.
    :param extra: optional additional data depending on ``what``.

    On exit the context manager writes a new document with fields as provided
    arguments, with these extra fields:

    * "result": one of "success", "model_error", "unhandled_error", where
      "model_error" means the wrapped code raised exception of type
      :exc:`genisys.web.errors.ModelError` and "unhandled_error" means
      any other kind of exception.
    * "exc_class": only if "result" is not "success". Contains string
      representation of the exception raised (like "Unauthorized").
    * "traceback": only if "result" is "unhandled_error". Contains string
      representation of the stack trace related to exception.
    """
    doc = {'who': who, 'who_groups': sorted(who_groups),
           'when': when, 'what': what,
           'path': path, 'revision': revision,
           'extra': _serialize(extra),
           'affected_rules': affected_rules}
    try:
        yield
    except errors.ModelError as exc:
        doc['result'] = 'model_error'
        doc['exc_class'] = type(exc).__name__
        database.audit.insert_one(doc)
        raise
    except Exception as exc:
        doc['result'] = 'unhandled_error'
        doc['exc_class'] = type(exc).__name__
        doc['traceback'] = traceback.format_exc()
        database.audit.insert_one(doc)
        raise
    else:
        doc['result'] = 'success'
        # assume that after successfull editing operation the revision
        # of the section has incremented. we want the new value to be
        # saved in the audit record, so the links from history page
        # would lead to what section had become, not to what it was.
        doc['revision'] += 1
        database.audit.insert_one(doc)


class BaseGenisysMongoStorage(object):
    def __init__(self, client, db_name):
        self.db = client[db_name]
        self.client = client
        self._create_indexes()

    @classmethod
    def _get_ts(cls):
        return time.time()

    @classmethod
    def _generate_oid(cls):
        return uuid.uuid4().hex

    @classmethod
    def _check_oid(cls, oid):
        return re.match(r'^[a-fA-F0-9]{32}$', oid) is not None

    def _create_indexes(self):
        self.db.volatile.create_indexes([
            pymongo.IndexModel([('vtype', pymongo.ASCENDING),
                                ('key', pymongo.ASCENDING)],
                               unique=True),
            pymongo.IndexModel([('locked', pymongo.ASCENDING),
                                ('etime', pymongo.ASCENDING)]),
        ])

    def _ensure_volatile(self, vtype, key, source, meta, now,
                         force_expire=False):
        hashed_key = volatile_key_hash(key)
        source = _serialize(source)
        while True:
            # first try to update only meta in existing record
            # with the same source value
            update = {"meta": meta}
            if force_expire:
                update['etime'] = now
                update['pcount'] = 0
            result = self.db.volatile.update_one(
                {"vtype": vtype, "key": hashed_key, "source": source},
                {"$set": update}
            )
            if result.matched_count != 0:
                # there was existing record with the same "source" value,
                # nothing more to do here
                break

            # no matched record found -- try to update record only by key.
            # in this case we forcibly unlock the record because even if
            # the record is currently being processed we do not want the
            # result of calculation on top of old source to be saved.
            # instead we want the record to be processed asap, that's why
            # we also set etime to now. we also reset counters, last_status
            # and ctime, since we consider volatile with different source
            # to be radically different -- no sense in continuing stats
            result = self.db.volatile.update_one(
                {"vtype": vtype, "key": hashed_key},
                {"$set": {
                    "locked": False,
                    "lock_id": None,
                    "ctime": now,
                    "etime": now,
                    "source": source,
                    "meta": meta,
                    "tcount": 0,
                    "ucount": 0,
                    "mcount": 0,
                    "pcount": 0,
                    "ecount": 0,
                    "last_status": "new",
                }}
            )
            if result.matched_count != 0:
                # there was existing record with different "source" value,
                # we have updated the record so that the older calculated
                # value gets invalidated. we also set "etime" so that the
                # record would be updated asap. well done!
                break

            try:
                self.db.volatile.insert_one({
                    "vtype": vtype,
                    "key": hashed_key,
                    "raw_key": key,
                    "locked": False,
                    "lock_id": None,
                    # creation / source changed time
                    "ctime": now,
                    # expiration time (when the value has to be updated next)
                    "etime": now,
                    # access time (last time volatile accessed for reading)
                    "atime": now,
                    # modified time (last time calculated value changed)
                    "mtime": None,
                    # update time (last time value calculated successfully)
                    "utime": None,
                    # touched time (last attempt to calculate value performed)
                    "ttime": None,
                    # touched count (number of attempts to update performed
                    # since last source change)
                    "tcount": 0,
                    # updated count (number of times value successfully
                    # calculated since last source change)
                    "ucount": 0,
                    # modified count (number of times calculated value changed
                    # since last source change)
                    "mcount": 0,
                    # number of times the processing was postponed since the
                    # last succesfull calculation (or since record creation).
                    "pcount": 0,
                    # number of times the processing failed since the last
                    # succesfull calculation (or since record creation).
                    "ecount": 0,
                    # one of "new", "error", "postponed", "same", "modified"
                    "last_status": "new",
                    "proclog": [],
                    "source": source,
                    "meta": meta,
                    "value": None,
                })
                # successfully inserted a new volatile record
                break
            except pymongo.errors.DuplicateKeyError:
                # somehow we failed to update existing record, but could
                # not insert a new one either. could be the record
                # was inserted elsewhere immediately after our update queries.
                # let's repeat the process all over again.
                continue

    def _base_get_volatiles(self, vtype, keys, projection=None):
        keys = list(keys)
        hashed_keys = [volatile_key_hash(key) for key in keys]
        key_by_hash = {hashed_key: keys[i]
                       for i, hashed_key in enumerate(hashed_keys)}
        records = list(self.db.volatile.find({"vtype": vtype,
                                              "key": {"$in": hashed_keys}},
                                             projection))
        result = {}
        for record in records:
            if record.get('value') is not None:
                record['value'] = _deserialize(record['value'])
            if record.get('source') is not None:
                record['source'] = _deserialize(record['source'])
            result[key_by_hash[record['key']]] = record

        return result

    def get_volatiles(self, vtype, keys, strict=True, full=False):
        """
        Get a dictionary of volatile records of given ``vtype``.

        :param vtype: string, volatile type.
        :param keys: sequence of unique identifiers of volatile records
            within ``vtype`` (unhashed).
        :param strict: make sure all values retrieved.
        :param full: include also (possibly heavy) values for keys
            'source' and 'value'.
        :returns: dict with keys being items from ``keys`` and values being
            volatile records.
        :raises genisys.web.errors.NotFound: if ``strict`` is True
            and at least one record could not be found.
        """
        projection = None
        if not full:
            projection = {'source': False, 'value': False, '_id': False}
        keys = set(keys)
        recs = self._base_get_volatiles(vtype, keys, projection)
        if strict and keys != set(recs):
            raise errors.NotFound()
        return recs

    def get_volatile(self, vtype, key_hash, full=False):
        """
        Get a single volatile record by its key hash.

        :param vtype: string, volatile type.
        :param keys: hashed unique identifier of a volatile record.
        :param full: include also (possibly heavy) values for keys
            'source' and 'value'.
        :returns: volatile record dictionary.
        :raises genisys.web.errors.NotFound: if such volatile record
            could not be found.
        """
        fltr = {'vtype': vtype, 'key': key_hash}
        projection = {'_id': False}
        if not full:
            projection['value'] = False
            projection['source'] = False
        result = self.db.volatile.find_one(fltr, projection)
        if result is None:
            raise errors.NotFound()
        if full:
            result['source'] = _deserialize(result['source'])
            if result['value']:
                result['value'] = _deserialize(result['value'])
        return result

    def force_volatile_update(self, vtype, key_hash):
        """
        Make volatile get updated as soon as possible.

        :param vtype: string, volatile type.
        :param key: hash of unique identifier of the volatile within ``vtype``.
        :returns: current timestamp to be used as ``not_earlier_than``
            argument in :meth:`get_updated_volatile`.
        :raises genisys.web.errors.NotFound: if no such volatile exists.
        """
        # use integer timestamp here so that we don't overwhelm supposed
        # clients (who will call :meth:`get_updated_volatile` later on)
        # with floating point values. since we don't round but rather
        # drop fractional part, this integer timestamp is in the past anyway
        now = int(self._get_ts())
        result = self.db.volatile.update_one(
            {"vtype": vtype, "key": key_hash},
            {"$set": {"locked": False, "lock_id": None, "etime": now}}
        )
        if result.matched_count == 0:
            raise errors.NotFound()
        return now

    def get_updated_volatile(self, vtype, key_hash, not_earlier_than):
        """
        Get volatile record only if it was updated recently.

        :param vtype: string, volatile type.
        :param key: hash of unique identifier of the volatile within ``vtype``.
        :param not_earlier_than: timestamp to compare updated date with.
        :returns: ``None`` if no such volatile could be found or if requested
            volatile record was not updated since ``not_earlier_than``.
            Otherwise -- a single volatile record.
        """
        result = self.db.volatile.find_one(
            {'vtype': vtype, 'key': key_hash,
             'utime': {'$gt': not_earlier_than}},
            {'_id': False}
        )
        if result is None:
            return None
        result['source'] = _deserialize(result['source'])
        if result['value']:
            result['value'] = _deserialize(result['value'])
        return result


class GenisysWebStorage(BaseGenisysMongoStorage):
    def _create_indexes(self):
        super(GenisysWebStorage, self)._create_indexes()
        self.db.section.create_indexes([
            pymongo.IndexModel([('path', pymongo.ASCENDING),
                                ('revision', pymongo.DESCENDING)],
                               unique=True),
            pymongo.IndexModel([('owners', pymongo.ASCENDING)]),
            pymongo.IndexModel([('marked_by', pymongo.ASCENDING)]),
            pymongo.IndexModel([('all_editors', pymongo.ASCENDING)]),
        ])
        self.db.audit.create_indexes([
            pymongo.IndexModel([('path', pymongo.ASCENDING),
                                ('when', pymongo.DESCENDING)]),
        ])

    def init_db(self, root_owners):
        """
        Initialize storage, by creating root section, first audit record
        and indexes in both collections.

        Should only be called once per storage.

        :param root_owners: list of unsernames of owners of the root section.
        """
        now = self._get_ts()
        root_desc = "Root section"
        root_creator = root_owners[0]
        root_owners = sorted(root_owners)
        au = audit(self.db, who=root_creator, when=now, what="create_section",
                   who_groups=[],
                   path="", revision=1, affected_rules=[],
                   # preserve in "extra" the same signature as for ordinary
                   # sections, see :meth:`create_section`
                   extra={'owners': root_owners, "desc": root_desc})
        with au:
            root_section = {
                "_id": self._generate_oid(),
                "path": "",
                "revision": 1,
                "name": "genisys",
                "desc": root_desc,
                "owners": root_owners,
                "subsections": _serialize({}),
                "changed_by": root_creator,
                "marked_by": [],
                "all_editors": [],
                "rules": _serialize([]),
                "ctime": now,
                "mtime": now,
                "stype": 'yaml',
                "stype_options": None,
            }
            self.db.section.insert_one(root_section)
        root_section['rules'] = []
        self._ensure_section_volatiles(root_creator, root_section, now,
                                       reresolve_selectors=True)
        self._create_indexes()

    def _find_section(self, path, revision=None, projection=None):
        fltr = {'path': path}
        if revision is not None:
            fltr['revision'] = revision
        result = list(self.db.section.find(
            fltr, projection=projection, limit=1,
            sort=[('path', pymongo.ASCENDING),
                  ('revision', pymongo.DESCENDING)]
        ))
        if not result:
            raise errors.NotFound()
        [result] = result
        if projection is None or 'rules' in projection:
            result['rules'] = _deserialize(result['rules'])
        if projection is None or 'subsections' in projection:
            result['subsections'] = _deserialize(result['subsections'])
        return result

    def _get_sections(self, oids, projection=None):
        sections = self.db.section.find({'_id': {'$in': oids}}, projection)
        result = collections.OrderedDict()
        for section in sections.sort([('path', 1), ('revision', -1)]):
            if 'rules' in section:
                section['rules'] = _deserialize(section['rules'])
            if 'subsections' in section:
                section['subsections'] = _deserialize(section['subsections'])
            result[section['_id']] = section
        return result

    def _get_section_ancestors(self, path):
        root = self._find_section(path="")
        if path == "":
            return [root]
        all_oids = []
        subsections = root['subsections']
        for name in path.split('.'):
            ss = subsections.get(name)
            if ss is None:
                raise errors.NotFound()
            all_oids.append(ss['_id'])
            subsections = ss['subsections']
        return [root] + list(self._get_sections(all_oids).values())

    def _all_section_rules(self, section):
        """
        Generate pairs (parent_rule, rule) from a full list of section rules.
        parent_rule is None for root rules.
        """
        for rule in section['rules']:
            yield None, rule
            for subrule in rule['subrules']:
                yield rule, subrule

    def _find_rule(self, rules, rulename, only_toplevel=False):
        for rule in rules:
            if rule['name'] == rulename:
                return rule, None
            if only_toplevel:
                continue
            for subrule in rule['subrules']:
                if subrule['name'] == rulename:
                    return subrule, rule
        raise errors.NotFound('rule {} not found'.format(rulename))

    def is_alive(self):
        """
        Check if this storage instance has a connection to a database server
        and that server is writeable (primary in replica set, master in
        master/slave configuration or standalone).
        """
        try:
            locked = self.client.is_locked
        except pymongo.errors.PyMongoError:
            return False
        return (not locked) and self.client.is_primary

    @classmethod
    def normalize_userlist(cls, userlist):
        """
        Remove duplicates from userlist, put all groups in alphabetical order
        in front and all logins in alphabetical order after.
        """
        u_users = set()
        u_groups = set()
        for u in userlist:
            if u.startswith('group:'):
                u_groups.add(u)
            else:
                u_users.add(u)
        return sorted(u_groups) + sorted(u_users)

    @classmethod
    def is_user_in_userlist(cls, username, usergroups, userlist):
        """
        Return True if user with login ``username``, belonging to groups
        ``usergroups`` is mentioned in a list ``userlist`` directly
        or as a group member.
        """
        u_users = set()
        u_groups = set()
        for u in userlist:
            if u.startswith('group:'):
                u_groups.add(u[len('group:'):])
            else:
                u_users.add(u)
        if username in u_users:
            return True
        return bool(set(usergroups).intersection(u_groups))

    def get_all_owners(self, path):
        """
        Get a set of usernames of all (own and inherited) owners
        of last revision of section with path ``path``.
        """
        root = self._find_section(path="", projection={
            'subsections': True, 'owners': True, '_id': False
        })
        all_owners = set(root['owners'])
        if path != "" and root['subsections']:
            all_oids = []
            subsections = root['subsections']
            for name in path.split('.'):
                all_oids.append(subsections[name]['_id'])
                subsections = subsections[name]['subsections']
            for rec in self.db.section.find({"_id": {"$in": all_oids}},
                                            {"owners": True, "_id": False}):
                all_owners.update(rec['owners'])
        return self.normalize_userlist(all_owners)

    def is_owner(self, username, usergroups, path):
        """
        Return True if user with login ``username``, belonging to groups
        in a list ``usergroups``, has direct or inherited owner rights
        on the last revision of section with path ``path``.
        """
        all_owners = self.get_all_owners(path)
        return self.is_user_in_userlist(username, usergroups, all_owners)

    def is_section_name_valid(self, name):
        """
        Check section name for validity.

        :returns: True if ``name`` is valid section name, otherwise False.
        """
        # http://docs.mongodb.org/manual/reference/limits/#Restrictions-on-Field-Names
        if not name or not isinstance(name, str):
            return False
        if '\0' in name or '.' in name or name[0] == '$':
            return False
        return True

    def create_section(self, username, usergroups, parent_path, parent_rev,
                       name, desc, owners, stype, stype_options):
        """
        Create a subsection in an existing section.

        :param username: name of the user who tries to create a section. User
            must be in the owners list of parent section's ancestors chain.
        :param usergroups: list of groups the user belongs to.
        :param parent_path: path of existing section, which should become
            a parent of the newly created section.
        :param parent_rev: revision number of the parent section.
        :param name: name of the newly created section, must comply with rules
            in :meth:`is_section_name_valid`.
        :param desc: string description of the section.
        :param owners: list of usernames of owners of new section.
        :param stype: section type, defining rules config type. Things like
            "yaml", or "sandbox_resource".
        :param stype_options: options, specific to stype.

        :raises ValueError:
            in case ``name`` is not valid section name.
        :raises genisys.web.errors.NotFound:
            if section with path ``parent_path`` doesn't exist.
        :raises genisys.web.errors.Outdated:
            if section with path ``parent_path`` has revision older than
            ``parent_rev``.
        :raises genisys.web.errors.Unauthorized:
            if ``username`` is not one of parent sections' owner.
        :raises genisys.web.errors.NotUnique:
            if section with path ``parent_path`` already contains child
            section with name ``name``.

        :returns: path of new section.

        Sections are being stored in a tree structure with parents referencing
        children. Each section document has field "subsections", wich is
        a mapping of children names to respective section document _ids.

        Each section stores in field "path" a materialized path to it in the
        section hierarchy. Path components are names of ancestors (including
        section itself) and delimiter is dot ("."), which is not valid
        character in a section name. Root section has path "" and all its
        first-level children have paths without dot (just the same as name).

        Each section has a revision number. Fields "path" and "revision"
        together form an unique index.
        """
        if not self.is_section_name_valid(name):
            raise ValueError("invalid section name '{}'".format(name))
        owners = self.normalize_userlist(owners)
        if parent_path == "":
            path = name
        else:
            path = ".".join((parent_path, name))

        now = self._get_ts()
        section = {
            "path": path,
            "name": name,
            "desc": desc,
            "owners": owners,
            "subsections": {},
            "changed_by": username,
            "marked_by": [],
            "all_editors": [],
            "rules": [],
            "ctime": now,
            "mtime": now,
            "stype": stype,
            "stype_options": stype_options,
        }
        extra = {'owners': owners, 'desc': desc, 'stype': stype,
                 'stype_options': stype_options}
        is_owner = self.is_owner(username, usergroups, parent_path)
        self._save_section(username, usergroups, section, parent_rev,
                           action='create_section', affected_rules=[],
                           extra=extra, now=now,
                           authorized=is_owner,
                           reresolve_selectors=True)
        return path

    def delete_empty_section(self, username, usergroups,
                             section_path, old_revision):
        """
        Delete a section with no subsections and no rules.

        :param username: name of the user who tries to delete a section. User
            must be section owner.
        :param usergroups: list of groups the user belongs to.
        :param section_path: path of the section to delete.
        :param old_revision: current revision of the section.

        :raises ValueError:
            if ``section_path`` is an empty string (represents path
            of the root section).
        :raises genisys.web.errors.NotFound:
            if section with path ``section_path`` doesn't exist.
        :raises genisys.web.errors.Outdated:
            if ``old_revision`` is not current revision of section.
        :raises genisys.web.errors.Unauthorized:
            if ``username`` is not one of section's ancestor chain owner.
        """
        if not section_path:
            raise ValueError("deleting root section is not allowed")
        section = self._find_section(section_path, old_revision)
        extra = section.copy()
        del extra['_id']
        section['deleted'] = True
        authorized = self.is_owner(username, usergroups, section_path) and \
            not section['rules'] and not section['subsections']
        self._save_section(username, usergroups, section, old_revision,
                           action='delete_empty_section',
                           affected_rules=[rule['name']
                                           for rule in section['rules']],
                           extra=extra, now=self._get_ts(),
                           authorized=authorized, reresolve_selectors=False)

    def set_section_desc(self, username, usergroups,
                         section_path, old_revision, new_desc):
        """
        Change section description text.

        :param username: name of the user who tries to change description.
            Must be one of section owners.
        :param usergroups: list of groups the user belongs to.
        :param section_path: path of the section to update.
        :param old_revision: current revision of the section.
        :param new_desc: new description string.

        :raises genisys.web.errors.NotFound:
            if section with path ``section_path`` doesn't exist.
        :raises genisys.web.errors.Outdated:
            if ``old_revision`` is not current revision of section.
        :raises genisys.web.errors.Unauthorized:
            if ``username`` is not one of section's ancestor chain owner.
        """
        section = self._find_section(section_path, old_revision)
        extra = {'prev': section['desc'], 'new': new_desc}
        section['desc'] = new_desc
        self._save_section(username, usergroups, section, old_revision,
                           action='set_section_desc', affected_rules=[],
                           extra=extra, now=self._get_ts(),
                           reresolve_selectors=False)

    def set_section_owners(self, username, usergroups,
                           section_path, old_revision, new_owners):
        """
        Change section owners list.

        :param username: name of the user who tries to change owners.
            Must be in the owners list.
        :param usergroups: list of groups the user belongs to.
        :param section_path: path of the section to update.
        :param old_revision: current revision of the section.
        :param new_owners: new owners list. ``username`` gets included
            implicitly if ``section_path`` is "" (root section).

        :raises genisys.web.errors.NotFound:
            if section with path ``section_path`` doesn't exist.
        :raises genisys.web.errors.Outdated:
            if ``old_revision`` is not current revision of section.
        :raises genisys.web.errors.Unauthorized:
            if ``username`` is not one of previous section's ancestor
            chain owner.
        """
        section = self._find_section(section_path, old_revision)
        new_owners = set(new_owners)
        if section_path == '':
            # one's not allowed to remove himself from root owners list
            new_owners.add(username)
        new_owners = self.normalize_userlist(new_owners)
        extra = {'prev': section['owners'], 'new': new_owners}
        authorized = self.is_owner(username, usergroups, section_path)
        section['owners'] = new_owners
        self._save_section(username, usergroups, section, old_revision,
                           action='set_section_owners', affected_rules=[],
                           extra=extra, now=self._get_ts(),
                           authorized=authorized, reresolve_selectors=False)

    def get_section_subtree(self, root_path, revision=None, max_depth=-1,
                            structure_only=False):
        """
        Get recursive dict structure, representing section subtree, starting
        from section with path ``root_path`` and including it.

        :param root_path: path of root section of tree to return.
            Should be "" for returning the full tree.
        :param revision: revision number of section with specified path.
            If not provided, the last revision is being used.
        :param max_depth: maximum depth of tree to request. The whole tree
            is being fetched if ``max_depth`` is negative.
        :param structure_only: if True dicts in result tree will only contain
            keys "name", "path" and subsections.
        :return: dict with fields of section document, with "subsections"
            field being replaced by a mapping of child names to child dicts.

        :raises genisys.web.errors.NotFound:
            if section with path ``root_path`` and revision ``revision``
            doesn't exist.

        .. note:: there is no auth check, all the data in section collection
            is world-readable.
        """
        projection = None
        if structure_only:
            projection = {'name': True, 'path': True, 'subsections': True}
        root = self._find_section(path=root_path, revision=revision,
                                  projection=projection)

        if structure_only:
            stack = [((root['path'], ), root)]
            while stack:
                path, sec = stack.pop()
                del sec['_id']
                if max_depth >= 0 and len(path) - 1 >= max_depth:
                    sec['subsections'] = sorted(sec['subsections'].keys())
                    continue
                for subsec_name, subsec_info in sec['subsections'].items():
                    subsec_info['name'] = subsec_name
                    subsec_path = path + (subsec_info['name'], )
                    subsec_info['path'] = '.'.join(subsec_path).lstrip('.')
                    stack.append((subsec_path, subsec_info))
            return root

        stack = [(0, (root['path'], ), root)]
        all_oids = []
        all_paths = []
        while stack:
            depth, path, sec = stack.pop()
            all_paths.append('.'.join(path).lstrip('.'))
            if max_depth >= 0 and depth >= max_depth:
                continue
            for subsec_name, subsec_info in sec['subsections'].items():
                all_oids.append(subsec_info['_id'])
                stack.append((depth + 1, path + (subsec_name, ), subsec_info))

        if not all_oids:
            sections = {}
        else:
            sections = self._get_sections(all_oids, projection)

        statkeys = ('mtime etime ctime utime ttime mcount ucount tcount '
                    'pcount ecount '
                    'last_status locked key raw_key vtype'.split())
        projection = {key: True for key in statkeys}
        volstat = self._base_get_volatiles(vtype='section', keys=all_paths,
                                           projection=projection)

        stack = [(0, (root['path'], ), root)]
        while stack:
            depth, path, sec = stack.pop()
            ppath = '.'.join(path).lstrip('.')
            sec['status'] = volstat.get(ppath)
            if sec['status']:
                del sec['status']['_id']
                del sec['status']['key']
            del sec['_id']
            if max_depth >= 0 and depth >= max_depth:
                sec['subsections'] = sorted(sec['subsections'].keys())
                continue
            for subsec_info in sec['subsections'].values():
                subsec = sections[subsec_info['_id']]
                sec['subsections'][subsec['name']] = subsec
                stack.append((depth + 1, path + (subsec['name'], ), subsec))
        return root

    def _save_rules(self, username, usergroups, section, new_rules, action,
                    affected_rules, extra,
                    authorized=None, reresolve_selectors=True):
        """
        Create a new revision for section with different list of rules.

        :param new_rules: new rules list.
        :param affected_rules: list of rule names affected by this change.
            Only used as extra info in audit record.

        Most of parameters are passed directly to :meth:`_save_section`.
        """
        now = self._get_ts()
        prev_rules_names = set(rule['name'] for rule in section['rules'])
        for rule in section['rules']:
            prev_rules_names.update(subrule['name']
                                    for subrule in rule['subrules'])
        new_rule_names = []
        for prule in new_rules:
            for rule in [prule] + prule['subrules']:
                if rule['name'] in affected_rules:
                    rule['mtime'] = now
                if rule['name'] not in prev_rules_names:
                    rule['ctime'] = now
                new_rule_names.append(rule['name'])
        if len(new_rule_names) != len(set(new_rule_names)):
            raise errors.NotUnique()
        extra.update({'prev_rules': section['rules']})
        section['rules'] = new_rules
        self._save_section(username, usergroups, section, section['revision'],
                           action, affected_rules, extra, now, authorized,
                           reresolve_selectors=reresolve_selectors)

    def save_section_resource_aliases(self, username, usergroups, path,
                                      old_revision, aliases):
        now = self._get_ts()
        section = self._find_section(path, old_revision)
        if section['stype'] != 'sandbox_resource':
            raise errors.Unauthorized('invalid section stype')
        old_aliases_by_id = {
            alias['id']: alias
            for alias in section['stype_options'].get('aliases', [])
        }
        new_aliases = []
        new_aliases_by_id = {}
        changed_aliases = set()

        for alias in aliases:
            if not alias.get('id'):
                alias['id'] = self._generate_oid()
            else:
                if not self._check_oid(alias['id'], ):
                    raise errors.Unauthorized('invalid alias id %r' %
                                              (alias['id'], ))
            if not alias.get('name') or not alias.get('resource') \
                    or not alias.get('description'):
                raise errors.Unauthorized('invalid alias %r' % (alias, ))
            alias = {
                'id': alias['id'],
                'name': alias['name'],
                'resource_id': alias['resource'],
                'resource_description': alias['description'],
            }
            old_alias = old_aliases_by_id.get(alias['id'])
            if old_alias != alias:
                changed_aliases.add(alias['id'])
            new_aliases.append(alias)
            if alias['id'] in new_aliases_by_id:
                raise errors.Unauthorized('alias id %r is not unique' %
                                          (alias['id'], ))
            new_aliases_by_id[alias['id']] = alias

        for alias_id in old_aliases_by_id:
            if alias_id not in new_aliases_by_id:
                changed_aliases.add(alias_id)

        section['stype_options']['aliases'] = new_aliases
        extra = {'prev': old_aliases_by_id, 'new': new_aliases_by_id}

        affected_rules = []
        for _, rule in self._all_section_rules(section):
            if rule['config_source']['rtype'] != 'by_alias':
                continue
            alias_id = rule['config_source']['alias_id']
            alias = new_aliases_by_id.get(alias_id)
            if not alias:
                raise errors.Unauthorized('alias %r is missing' % (alias_id, ))
            rule['config'] = {'resource_id': alias['resource_id']}
            rule['config_source'] = {
                'rtype': 'by_alias',
                'alias_id': alias_id,
                'alias_name': alias['name'],
                'resource': alias['resource_id'],
                'description': alias['resource_description'],
            }
            rule['mtime'] = now
            if alias_id in changed_aliases:
                affected_rules.append(rule['name'])

        self._save_section(username, usergroups, section, old_revision,
                           action='save_aliases',
                           affected_rules=affected_rules,
                           extra=extra, now=now)

    def _save_section(self, username, usergroups, section, old_revision,
                      action, affected_rules, extra, now,
                      authorized=None, reresolve_selectors=True):
        """
        Create a new revision of a section as well as all of its ancestors.

        :param username: name of the user who saves a section.
        :param usergroups: list of group names user belongs to.
        :param section: a section dictionary to save. if key 'revision'
            is missing from that dict, the section is considered to be new.
            if key 'deleted' is present and is true, the section is considered
            to be deleted.
        :param old_revision: section's revision number before update. If the
            section is being created (``section`` dict has no 'revision') --
            this is a revision of parent section.
        :param action: string literal describing the change made, see
            :func:`audit`, parameter ``what``.
        :param affected_rules: list of rule names affected by the change.
        :param extra: dict of additinal data to attach to audit record.
        :param now: current timestamp.
        :param authorized: If ``True``, assume user has rights to make
            the change (permission has been checked outside
            of :func:`_save_section`). If ``False`` -- just raise an exception
            :exc:`~genisys.web.errors.Unauthorized` and store an audit record,
            don't save anything else. If ``None`` (default) -- check that
            the user is in the owners list of the section's ancestor chain,
            raise :exc:`~genisys.web.errors.Unauthorized` otherwise.

        :raises genisys.web.errors.Unauthorized:
            if ``authorized`` is False, or ``authorized`` is None
            and ``username`` is not one of section's own/inherited owners.
        :raises genisys.web.errors.Outdated:
            if revision ``old_revision`` is not the latest for section.
        :raises genisys.web.errors.NotUnique:
            if names of rules are not unique or if new section has the same
            name as one of existing siblings.
        """
        # get a list of section ancestors, including section itself.
        # it's important that the list goes from bottom to top,
        # so that the first item is deepmost section and the last
        # one is root section. we create replacement section documents
        # in the same order, so the next revision of the root section
        # gets inserted last, and when it happened, the new revision
        # of whole tree is established.
        if 'revision' not in section:
            # creating new section
            if '.' not in section['path']:
                parent_path = ''
            else:
                parent_path, _ = section['path'].rsplit('.', 1)
            revision_path = parent_path
            sections = self._get_section_ancestors(parent_path)
            section['revision'] = sections[0]['revision']
            real_old_revision = sections[-1]['revision']
            sections.append(section)
            if section['name'] in sections[-2]['subsections']:
                raise errors.NotUnique(
                    "name '{}' is not unique within section '{}'".format(
                        section['name'], parent_path
                    )
                )
        else:
            # editing existing section
            sections = self._get_section_ancestors(section['path'])
            real_old_revision = sections[-1]['revision']
            sections[-1].update({
                'desc': section['desc'],
                'owners': section['owners'],
                'rules': section['rules'],
                'stype': section['stype'],
                'stype_options': section['stype_options'],
                'changed_by': username,
            })
            if section.get('deleted'):
                sections[-1]['deleted'] = True
            revision_path = section['path']
        sections.reverse()
        sections[0]['mtime'] = now
        sections[0]['owners'] = self.normalize_userlist(section['owners'])
        sections[0]['all_editors'] = self.normalize_userlist(itertools.chain(*(
            rule['editors'] for _, rule in self._all_section_rules(section)
        )))
        section = sections[0].copy()
        if old_revision != real_old_revision:
            raise errors.Outdated(
                "current revision of section '{}' is {}, not {}".format(
                    revision_path, real_old_revision, old_revision
                )
            )

        au = audit(self.db, who=username, who_groups=usergroups,
                   when=now, what=action,
                   path=section['path'], revision=section['revision'],
                   extra=extra, affected_rules=affected_rules)
        with au:
            if authorized is None:
                if not self.is_owner(username, usergroups, section['path']):
                    raise errors.Unauthorized()
            elif authorized is False:
                raise errors.Unauthorized()
            elif authorized is True:
                pass

            new_oids = [self._generate_oid() for _ in sections]
            for i, sec in enumerate(sections):
                sec['_id'] = new_oids[i]
                sec['ctime'] = now
                sec['revision'] += 1

                if i == 0:
                    pass
                elif i == 1 and section.get('deleted'):
                    del sections[1]['subsections'][section['name']]
                else:
                    sec['subsections'][sections[i - 1]['name']] = {
                        '_id': new_oids[i - 1],
                        'subsections': sections[i - 1]['subsections']
                    }

            for sec in sections:
                sec['rules'] = _serialize(sec['rules'])
                sec['subsections'] = _serialize(sec['subsections'])

            try:
                self.db.section.insert_many(sections, ordered=True)
            except pymongo.errors.BulkWriteError as exc:
                # got a concurent write? "reverting" "transaction"
                self.db.section.delete_many({'_id': {'$in': new_oids}})
                err_msg = '; '.join(werr['errmsg']
                                    for werr in exc.details['writeErrors'])
                raise errors.Outdated(err_msg)

            section['revision'] += 1
            self._ensure_section_volatiles(username, section, now,
                                           reresolve_selectors)

    def _ensure_section_volatiles(self, username, section, now,
                                  reresolve_selectors):
        all_rules = []
        all_selectors = set()
        for rule in section['rules']:
            selector_keys = tuple()
            if rule['selector'] is not None:
                selector_keys = (volatile_key_hash(rule['selector']), )
                all_selectors.add(rule['selector'])
            for subrule in rule['subrules']:
                subrule_selector_keys = selector_keys
                if subrule['selector'] is not None:
                    subrule_selector_keys = selector_keys + (volatile_key_hash(
                        subrule['selector']
                    ), )
                    all_selectors.add(subrule['selector'])
                all_rules.append((subrule, rule, subrule_selector_keys))
            all_rules.append((rule, None, selector_keys))

        # update selectors from all the rules
        for selector in all_selectors:
            self._ensure_volatile(vtype="selector", key=selector,
                                  source=selector, now=now, meta={},
                                  force_expire=reresolve_selectors)

        source_extra = {}
        if section['stype'] == 'sandbox_resource':
            resource_type = section['stype_options']['resource_type']
            self._ensure_volatile(
                vtype="sandbox_releases", key=resource_type, now=now,
                source={'resource_type': resource_type}, meta={},
            )
            for rule, parent_rule, _ in all_rules:
                # for sandbox sections -- update resource info
                # for all the resources selected in rules
                resource_id = rule['config']['resource_id']
                self._ensure_volatile(vtype='sandbox_resource',
                                      key=resource_id, source=rule['config'],
                                      now=now, meta={}, force_expire=False)
            # pre-calculate needed hashed keys
            section_volatile_rules_list = [
                {'rule_name': rule['name'],
                 'selector_keys': selector_keys,
                 'resource_key': volatile_key_hash(
                     rule['config']['resource_id']
                 )}
                for rule, _, selector_keys in all_rules
            ]
            # we need this key only to update its atime when we process
            # the section -- that is the only place where we do it,
            # so the releases list is being accessed (and therefore exists)
            # for as long as a section referring it gets updated
            source_extra['sandbox_releases_key'] = volatile_key_hash(
                section['stype_options']['resource_type']
            )
        else:
            section_volatile_rules_list = [
                {'rule_name': rule['name'],
                 'selector_keys': selector_keys,
                 'config': rule['config']}
                for rule, _, selector_keys in all_rules
            ]
        if not section.get('deleted'):
            # update section record itself
            self._ensure_volatile(
                vtype="section", key=section['path'], now=now,
                source=dict(rules=section_volatile_rules_list,
                            stype=section['stype'],
                            stype_options=section['stype_options'],
                            **source_extra),
                meta={'revision': section['revision'],
                      'changed_by': username,
                      'mtime': now,
                      'owners': section['owners'],
                      'reresolve_selectors': reresolve_selectors},
                force_expire=True
            )
        else:
            # section was just deleted: removing record from volatile
            # collection
            self.db.volatile.delete_one(
                {'vtype': 'section', 'key': volatile_key_hash(section['path'])}
            )

    def create_rule(self, username, usergroups,
                    path, old_revision, parent_rule,
                    rulename, desc, editors, selector, config, config_source):
        """
        Create a new rule or subrule.

        :param username: name of the user who tries to create a rule.
            Must be in the ancestor chain owners list.
        :param usergroups: list of groups the user belongs to.
        :param path: path of the section to update.
        :param old_revision: section's revision number before update.
        :param parent_rule: name of the parent rule to create a subrule within.
            May be ``None`` in which case a top-level rule is being created.
        :param rulename: string, name of the new rule. Must be unique
            within a section (including all subrules).
        :param desc: string, section description.
        :param editors: list of rule editors (persons who are allowed
            to change rule's config without being section owners) usernames.
        :param selector: blinov calc expression string.
        :param config: rule configuration dictionary.
        :param config_source: configuration in a form it was created by user.

        Editor of the top-level rule is allowed to create subrules in it.
        Otherwise user must be section owner either direct or inherited.

        Newly created rule gets inserted last in the rule list.

        See :meth:`_save_rules` for a list of possible exceptions.
        """
        section = self._find_section(path, revision=old_revision)
        rule = {'name': rulename,
                'desc': desc,
                'editors': self.normalize_userlist(editors),
                'selector': selector,
                'config': config,
                'config_source': config_source}
        new_rules = copy.deepcopy(section['rules'])
        authorized = None
        if not parent_rule:
            rule['subrules'] = []
            new_rules += [rule]
            affected_rules = [rulename]
        else:
            affected_rules = [rulename, parent_rule]
            parent_rule, _ = self._find_rule(new_rules, parent_rule,
                                             only_toplevel=True)
            parent_rule['subrules'] += [rule]
            if self.is_user_in_userlist(username, usergroups,
                                        parent_rule['editors']):
                authorized = True
        extra = rule.copy()
        extra['stype'] = section['stype']
        extra['stype_options'] = section['stype_options']
        extra['parent_rule'] = parent_rule['name'] if parent_rule else None
        return self._save_rules(username, usergroups, section, new_rules,
                                action='create_rule', authorized=authorized,
                                affected_rules=affected_rules, extra=extra,
                                reresolve_selectors=True)

    def edit_rule(self, username, usergroups, path, old_revision,
                  rulename, desc, editors, selector):
        """
        Edit rule.

        :param username: name of the user who tries to change a rule.
            Must be in the ancestor chain owners list or be an editor
            of parent rule if ``rulename`` specifies subrule.
        :param usergroups: list of groups the user belongs to.
        :param path: path of the section to update.
        :param old_revision: section's revision number before update.
        :param rulename: string, name of the rule to edit. The actual name
            can not be changed.
        :param desc: string, section description.
        :param editors: new list of rule editors (persons who are allowed
            to change rule's config without being section owners) usernames.
        :param selector: blinov calc expression string.

        :raises genisys.web.errors.NotFound:
            if section doesn't contain rule with name ``rulename``.

        See :meth:`_save_rules` for a list of other possible exceptions.
        """
        section = self._find_section(path, revision=old_revision)
        new_rules = copy.deepcopy(section['rules'])
        rule, parent_rule = self._find_rule(new_rules, rulename)
        affected_rules = [rulename]
        authorized = None
        if parent_rule:
            if self.is_user_in_userlist(username, usergroups,
                                        parent_rule['editors']):
                authorized = True
            affected_rules.append(parent_rule['name'])
        reresolve_selectors = rule['selector'] != selector
        editors = self.normalize_userlist(editors)
        extra = {'prev': {'desc': rule['desc'], 'editors': rule['editors'],
                          'selector': rule['selector'], 'name': rulename},
                 'new': {'desc': desc, 'editors': editors,
                         'selector': selector, 'name': rulename}}
        rule.update({'desc': desc,
                     'editors': editors,
                     'selector': selector})
        return self._save_rules(username, usergroups, section, new_rules,
                                action='edit_rule', authorized=authorized,
                                affected_rules=affected_rules, extra=extra,
                                reresolve_selectors=reresolve_selectors)

    def edit_rule_config(self, username, usergroups, path, old_revision,
                         rulename, config, config_source):
        """
        Edit rule configuration dictionary.

        :param username: name of the user who tries to change config.
            Must be either in the ancestor chain owners list or in the rule
            (or parent rule) editors list.
        :param usergroups: list of groups the user belongs to.
        :param path: path of the section to update.
        :param old_revision: section's revision number before update.
        :param rulename: string, name of the rule to edit.
        :param config: rule configuration dictionary.
        :param config_source: configuration in a form it was created by user.

        :raises genisys.web.errors.NotFound:
            if section doesn't contain rule with name ``rulename``.

        See :meth:`_save_rules` for a list of other possible exceptions.
        """
        section = self._find_section(path, revision=old_revision)
        new_rules = copy.deepcopy(section['rules'])
        rule, parent_rule = self._find_rule(new_rules, rulename)
        affected_rules = [rulename]
        authorized = None
        if parent_rule:
            if self.is_user_in_userlist(username, usergroups,
                                        parent_rule['editors']):
                authorized = True
            affected_rules.append(parent_rule['name'])
        if self.is_user_in_userlist(username, usergroups, rule['editors']):
            authorized = True
        extra = {'prev': rule['config'],
                 'new': config,
                 'prev_source': rule['config_source'],
                 'new_source': config_source,
                 'stype': section['stype'],
                 'stype_options': section['stype_options']}
        rule.update({'config': config, 'config_source': config_source})
        return self._save_rules(username, usergroups, section, new_rules,
                                action='edit_rule_config',
                                authorized=authorized,
                                affected_rules=affected_rules, extra=extra,
                                reresolve_selectors=False)

    def reorder_rules(self, username, usergroups, path, old_revision,
                      parent_rule, new_order):
        """
        Change rules order within a section or subrules order within a rule.

        :param username: name of the user who tries to change order.
            Must be in the section's ancestor chain owners list for changing
            order of top-level rules or parent rule editor for changing order
            of subrules.
        :param usergroups: list of groups the user belongs to.
        :param path: path of the section to update.
        :param old_revision: section's revision number before update.
        :param parent_rule: name of the rule to change subrules order in
            or ``None`` for changing top-level rules order.
        :param new_order: list of integer rule (subrule) indexes.

        See :meth:`_save_rules` for a list of other possible exceptions.
        """
        section = self._find_section(path, revision=old_revision)
        authorized = None
        new_rules = copy.deepcopy(section['rules'])
        if parent_rule is not None:
            parent_rule, _ = self._find_rule(new_rules, parent_rule,
                                             only_toplevel=True)
            rules = parent_rule['subrules']
            if self.is_user_in_userlist(username, usergroups,
                                        parent_rule['editors']):
                authorized = True
        else:
            rules = new_rules

        if sorted(new_order) != list(range(len(rules))):
            raise errors.Unauthorized()

        newly_ordered = [rules[i] for i in new_order]
        extra = {'prev': [rule['name'] for rule in rules],
                 'new': [rule['name'] for rule in newly_ordered]}
        affected_rules = extra['new'].copy()
        if parent_rule is not None:
            affected_rules.append(parent_rule['name'])
        del rules[:]
        rules.extend(newly_ordered)
        return self._save_rules(username, usergroups, section, new_rules,
                                action='reorder_rules',
                                affected_rules=affected_rules,
                                authorized=authorized,
                                extra=extra, reresolve_selectors=False)

    def delete_rule(self, username, usergroups, path, old_revision, rulename):
        """
        Delete rule from section rules list or subrule from a top-level rule.

        :param username: name of the user who tries to delete rule.
            Must be in the section's ancestor chain owners list for deleting
            top-level rule or parent rule editor for deleting a subrule.
        :param usergroups: list of groups the user belongs to.
        :param path: path of the section to update.
        :param old_revision: section's revision number before update.
        :param rulename: string, name of the rule to delete.

        :raises genisys.web.errors.NotFound:
            if section doesn't contain rule with name ``rulename``.

        See :meth:`_save_rules` for a list of other possible exceptions.
        """
        section = self._find_section(path, revision=old_revision)
        new_rules = copy.deepcopy(section['rules'])
        rule, parent_rule = self._find_rule(new_rules, rulename)
        extra = rule.copy()
        extra['stype'] = section['stype']
        extra['stype_options'] = section['stype_options']
        affected_rules = [rulename]
        authorized = None
        if parent_rule is not None:
            rules = parent_rule['subrules']
            affected_rules.append(parent_rule['name'])
            if self.is_user_in_userlist(username, usergroups,
                                        parent_rule['editors']):
                authorized = True
        else:
            rules = new_rules
        rules_without_deleted = [rule for rule in rules
                                 if rule['name'] != rulename]
        del rules[:]
        rules.extend(rules_without_deleted)
        return self._save_rules(username, usergroups, section, new_rules,
                                action='delete_rule', authorized=authorized,
                                affected_rules=affected_rules, extra=extra,
                                reresolve_selectors=False)

    def revert_rules(self, username, usergroups,
                     path, current_rev, revert_to_rev):
        """
        Take rules from section's old revision and create a new one with it.

        Does the same as :meth:`save_rules`, but takes ``new_rules`` from
        the older revision ``revert_to_rev`` of the same section.
        """
        section = self._find_section(path, revision=current_rev)
        prev_rev = self._find_section(path, revert_to_rev,
                                      {'rules': True, 'stype': True,
                                       'stype_options': True})
        affected = set(rule['name'] for rule in section['rules'])
        affected.update(itertools.chain(*(
            (subrule['name'] for subrule in rule['subrules'])
            for rule in section['rules']
        )))
        affected.update(rule['name'] for rule in prev_rev['rules'])
        affected.update(itertools.chain(*(
            (subrule['name'] for subrule in rule['subrules'])
            for rule in prev_rev['rules']
        )))
        extra = {
            'prev_rev': section['revision'],
            'revert_to_rev': revert_to_rev,
            'prev': {'stype': section['stype'],
                     'stype_options': section['stype_options']},
            'new': {'stype': prev_rev['stype'],
                    'stype_options': prev_rev['stype_options']},
            'new_rules': prev_rev['rules'],
        }
        section['stype'] = prev_rev['stype']
        section['stype_options'] = prev_rev['stype_options']
        return self._save_rules(username, usergroups, section,
                                prev_rev['rules'], action='revert_rules',
                                affected_rules=sorted(affected),
                                extra=extra, reresolve_selectors=True)

    def get_section_history(self, path, offset=0, limit=50, recursive=False,
                            rule=None):
        """
        Get the time-reversed list of actions made with a section according
        to "audit" collection.

        See :func:`audit` for description of fields in list items.

        :param path: the path of the section to query history of. The section
            with such path does not have to exist now or have existed ever.
        :param offset: start output records from that index.
        :param limit: number of records to return.
        :param recursive: return history of all children of section with
            path ``path`` as well as its own history.
        :param rule: limit history to changes affecting this rule name only.
            Not compatible with ``recursive=True``.
        """
        if recursive:
            if path == "":
                # recursive history of root node means the entire history
                fltr = {}
            else:
                fltr = {'$or': [
                    {'path': {'$regex': '^{}\.'.format(re.escape(path))}},
                    {'path': path}
                ]}
        else:
            fltr = {'path': path}
        if not recursive and rule is not None:
            fltr['affected_rules'] = rule
        actions = self.db.audit.find(fltr, {'_id': False})
        actions = actions.sort([('when', pymongo.DESCENDING)])
        actions = list(actions.skip(offset).limit(limit))
        for action in actions:
            action['extra'] = _deserialize(action['extra'])
        return actions

    def get_dashboard(self, username, usergroups):
        """
        Get sections user has an ownership on, sections he marked and
        sections with rules he can edit.

        :param username: name of the user whose dashboard is being requested.
        :param usergroups: list of groups the user belongs to.
        :returns: dictionary with keys 'marked', 'owned' and 'editable'
            and values being sections itself (with its subsections).
        """
        usergroups = ['group:{}'.format(group) for group in usergroups]
        usergroups_set = set(usergroups)
        access_filter = {'$in': [username] + usergroups}
        roots = set(
            self.db.section.distinct('path', {'owners': access_filter})
        )
        if "" in roots:
            owned = self.get_section_subtree("")['subsections']
        else:
            dot_re = re.compile(r'\.')
            for path in list(roots):
                for match in dot_re.finditer(path):
                    superpath = path[:match.span()[0]]
                    if superpath in roots:
                        roots.discard(path)
                        break
            owned = {root: self.get_section_subtree(root)
                     for root in roots}
            # we need to check twice because the query above also selects
            # previous revisions, so the section might appear in the dashboard
            # as the one owned even if the user was removed from the owners
            # list in the latest revision
            for path, section in list(owned.items()):
                if username in section['owners']:
                    continue
                if usergroups_set.intersection(section['owners']):
                    continue
                del owned[path]
        marked = self.db.section.distinct('path', {'marked_by': username})
        marked = {path: self.get_section_subtree(path) for path in marked}
        editable = self.db.section.distinct('path',
                                            {'all_editors': access_filter})
        editable = {path: self.get_section_subtree(path) for path in editable}
        # see notice about double-check above
        for path, section in list(editable.items()):
            if username in section['all_editors']:
                continue
            if usergroups_set.intersection(section['all_editors']):
                continue
            del editable[path]
        return {'marked': marked, 'owned': owned, 'editable': editable}

    def mark_section(self, username, path):
        """
        Mark section as favorite by user.

        :param username: name of user who marks a section.
        :param path: section path to mark.
        """
        self.db.section.update_many({'path': path},
                                    {'$addToSet': {'marked_by': username}})

    def unmark_section(self, username, path):
        """
        Unmark section (remove from favorites).

        :param username: name of user who unmarks a section.
        :param path: section path to unmark.
        """
        self.db.section.update_many({'path': path},
                                    {'$pull': {'marked_by': username}})


def _deserialize(data):
    return msgpack.loads(codecs.decode(data, 'zip'), encoding='utf-8')


def _serialize(data):
    return codecs.encode(msgpack.dumps(data, encoding='utf-8'), 'zip')


def volatile_key_hash(val):
    if isinstance(val, bytes):
        pass
    elif isinstance(val, (list, tuple)) and val:
        val = ':'.join(map(volatile_key_hash, val)).encode('utf8')
    elif isinstance(val, (int, bool)):
        val = str(val).encode('utf8')
    elif isinstance(val, str):
        val = val.encode('utf8')
    else:
        raise ValueError('can not hash value of type {}'.format(type(val)))
    return hashlib.sha1(val).hexdigest()

VOLATILE_KEY_HASH_REGEXP = '[0-9a-f]{40}'


MongoStorage = GenisysWebStorage
