import logging
import functools

import requests
import flask


logger = logging.getLogger('genisys.sandbox')


class SandboxError(Exception):
    pass


def _sandbox_errors(func):
    @functools.wraps(func)
    def decorated(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except SandboxError:
            raise
        except Exception as exc:
            logger.exception(
                "happened on {}(*{}, **{})".format(func.__name__, args, kwargs)
            )
            error = "{}: {}".format(type(exc).__name__, exc)
            raise SandboxError(error)
    return decorated


def _sandbox_request(url, *args):
    flask.current_app.config
    url = flask.current_app.config['SANDBOX_URI'] + url
    resp = requests.get(url, *args, timeout=10)
    if resp.status_code == 200:
        return resp.json()
    if 'content-type' in resp.headers:
        if resp.headers['content-type'].startswith('application/json'):
            try:
                reason = resp.json()['reason']
            except:
                pass
            else:
                raise SandboxError(reason)
    resp.raise_for_status()


@_sandbox_errors
def get_resource_type_description(typename):
    [res] = _sandbox_request("/suggest/resource", {'type': typename})
    return res['description']


@_sandbox_errors
def get_resource_info(resource_id):
    return _sandbox_request("/resource/{}".format(resource_id))
