import flask
import statsd
import time


def set_up(app):
    app.stats = statsd.StatsClient(host=app.config['STATSD_HOST'],
                                   port=app.config['STATSD_PORT'])
    def before_request():
        flask.request.started_at = time.time()

    def teardown_request(exc):
        if exc is None:
            return
        try:
            collect_request_stats(status_code=500)
        except:
            pass

    def after_request(response):
        collect_request_stats(status_code=response.status_code)
        return response

    def collect_request_stats(status_code):
        endpoint = flask.request.endpoint
        method = flask.request.method.lower()
        signal = 'code.{} endpoint.{} method.{} http'.format(
            status_code, endpoint, method
        )
        time_spent = (time.time() - flask.request.started_at) * 1000
        app.stats.timing(signal, time_spent)
        user_agent = flask.request.user_agent.string.split(' ', 1)[0]
        if user_agent:
            user_agent = user_agent.replace('.', '-')
        else:
            user_agent = '(missing)'
        app.stats.timing('ua.{} {}'.format(user_agent, signal), time_spent)

    app.before_request(before_request)
    app.after_request(after_request)
    app.teardown_request(teardown_request)
