import itertools

import requests
import msgpack
import flask
from flask.ext.babel import gettext
from werkzeug.datastructures import MultiDict

from genisys.web import model, forms, errors


def _get_section(path, max_depth):
    revision = flask.request.args.get('revision')
    storage = flask.current_app.storage
    if revision is not None:
        try:
            revision = int(revision)
        except Exception:
            flask.abort(404)
        current_section = storage.get_section_subtree(path,
                                                      max_depth=max_depth)
        if current_section['revision'] == revision:
            section = current_section
        else:
            section = storage.get_section_subtree(path, max_depth=max_depth,
                                                  revision=revision)
    else:
        section = storage.get_section_subtree(path, max_depth=max_depth)
        current_section = section
    return section, current_section


def _get_released_resources(section):
    storage = flask.current_app.storage
    key = model.volatile_key_hash(section['stype_options']['resource_type'])
    volatile = storage.get_volatile('sandbox_releases', key, full=True)
    return volatile['value'] or []


def _get_all_selector_statuses(section, storage, full=False):
    all_selectors = set()
    for rule in section['rules']:
        if rule['selector']:
            all_selectors.add(rule['selector'])
        all_selectors.update(subrule['selector']
                             for subrule in rule['subrules']
                             if subrule['selector'])
    return storage.get_volatiles('selector', all_selectors, full=full)


def dashboard():
    dashboard = flask.current_app.storage.get_dashboard(
        flask.g.username, flask.g.usergroups
    )
    resource_types = set(section['stype_options']['resource_type']
                         for section in dashboard['editable'].values()
                         if section['stype'] == 'sandbox_resource')
    resources_by_type = {
        rtype: flask.current_app.storage.get_volatile(
            'sandbox_releases', model.volatile_key_hash(rtype), full=True
        )['value'] or []
        for rtype in resource_types
    }
    resources_forms = {}
    for section in dashboard['editable'].values():
        if not section['stype'] == 'sandbox_resource':
            continue
        rtype = section['stype_options']['resource_type']
        allrules = [[r] + [s for s in r['subrules']] for r in section['rules']]
        for rule in itertools.chain(*allrules):
            is_editor = flask.current_app.storage.is_user_in_userlist(
                flask.g.username, flask.g.usergroups, rule['editors']
            )
            if not is_editor:
                continue
            data = {'revision': section['revision'],  'action': 'edit-config'}
            form = forms.SandboxResourceRuleConfigForm(
                rtype, resources_by_type[rtype],
                aliases=section['stype_options'].get('aliases', []), data=data,
                config_source=rule['config_source']
            )
            resources_forms[(section['path'], rule['name'])] = form
    return flask.render_template('dashboard.html', marked=dashboard['marked'],
                                 owned=dashboard['owned'],
                                 editable=dashboard['editable'],
                                 resources_forms=resources_forms)


def mark_unmark_section(path):
    action = flask.request.form.get('action')
    if action == 'mark':
        flask.current_app.storage.mark_section(flask.g.username, path)
        flask.flash(gettext('Section marked as favorite'), 'success')
    elif action == 'unmark':
        flask.current_app.storage.unmark_section(flask.g.username, path)
        flask.flash(gettext('Section removed from favorites'), 'success')
    else:
        flask.abort(400)
    return flask.redirect(flask.url_for('section', path=path))


def section(path):
    action = flask.request.form.get('action')
    storage = flask.current_app.storage
    section, current_section = _get_section(path, max_depth=1)

    current_all_owners = storage.get_all_owners(path)
    if not current_section.get('deleted'):
        current_section['inherited_owners'] = storage.normalize_userlist(
            set(current_all_owners) - set(current_section['owners'])
        )
    if section['revision'] == current_section['revision'] \
            and not section.get('deleted'):
        section['inherited_owners'] = storage.normalize_userlist(
            set(current_all_owners) - set(section['owners'])
        )
    else:
        section['inherited_owners'] = []

    new_subsection_form = forms.NewSubsectionForm(data={
        'action': 'new-section',
        'parent_revision': section['revision']
    })
    desc_form = forms.DescriptionForm(
        data=dict(action='change-desc', **section)
    )
    owners_form = forms.OwnersForm(
        data=dict(action='change-owners', **section)
    )
    new_rule_data = {'revision': section['revision'], 'action': 'new-rule'}
    if section['stype'] == 'yaml':
        new_rule_form = forms.NewYamlRuleForm(data=new_rule_data)
    else:
        new_rule_form = forms.NewSandboxResourceRuleForm(
            resource_type=section['stype_options']['resource_type'],
            aliases=section['stype_options'].get('aliases', []),
            released_resources=_get_released_resources(section),
            data=new_rule_data
        )
    rule_order_form = forms.RuleOrderForm(data={
        'revision': section['revision'],
        'action': 'reorder-rules',
        'order': range(len(section['rules'])),
    })

    selector_statuses = _get_all_selector_statuses(section, storage)

    if action == 'new-section' and new_subsection_form.validate_on_submit():
        stype = new_subsection_form.stype.data
        stype_options = None
        if stype == new_subsection_form.STYPE_SANDBOX_RESOURCE:
            stype_options = {
                'resource_type': new_subsection_form.sandbox_resource_type.data
            }
        try:
            new_section_path = storage.create_section(
                flask.g.username, flask.g.usergroups, path,
                new_subsection_form.parent_revision.data,
                new_subsection_form.name.data, new_subsection_form.desc.data,
                new_subsection_form.owners.data,
                stype=stype, stype_options=stype_options
            )
        except errors.NotUnique:
            new_subsection_form['name'].errors.append(
                gettext('Subsection name is not unique.')
            )
        else:
            flask.flash(gettext('New subsection created'), 'success')
            return flask.redirect(flask.url_for('section',
                                                path=new_section_path))

    if action == 'new-rule' and new_rule_form.validate_on_submit():
        data = new_rule_form.data
        selector = (data['selector']
                    if data['htype'] == forms.EditRuleForm.HTYPE_SOME
                    else None)
        try:
            storage.create_rule(
                flask.g.username, flask.g.usergroups,
                path, old_revision=data['revision'],
                parent_rule=None, rulename=data['name'], desc=data['desc'],
                editors=data['editors'], selector=selector,
                config=new_rule_form.get_config(),
                config_source=new_rule_form.get_config_source(),
            )
        except errors.NotUnique:
            new_rule_form['name'].errors.append(
                gettext('Rule name is not unique.')
            )
        else:
            flask.flash(gettext('New rule created'), 'success')
            return flask.redirect(flask.url_for('rule', path=path,
                                                rulename=data['name']))

    if action == 'reorder-rules' and rule_order_form.validate_on_submit():
        old_revision = rule_order_form.data['revision']
        storage.reorder_rules(flask.g.username, flask.g.usergroups,
                              path, old_revision, parent_rule=None,
                              new_order=rule_order_form.data['order'])
        flask.flash(gettext('New rules order saved'), 'success')
        return flask.redirect(flask.url_for('section', path=path))

    if action == 'change-desc' and desc_form.validate_on_submit():
        storage.set_section_desc(flask.g.username, flask.g.usergroups, path,
                                 desc_form.revision.data, desc_form.desc.data)
        flask.flash(gettext('Section description saved'), 'success')
        return flask.redirect(flask.url_for('section', path=path))

    if action == 'change-owners' and owners_form.validate_on_submit():
        storage.set_section_owners(flask.g.username, flask.g.usergroups, path,
                                   owners_form.revision.data,
                                   owners_form.owners.data)
        flask.flash(gettext('Section owners saved'), 'success')
        return flask.redirect(flask.url_for('section', path=path))

    is_owner = storage.is_user_in_userlist(
        flask.g.username, flask.g.usergroups,
        section['owners'] + section['inherited_owners']
    )
    return flask.render_template(
        'section.html', section=section,
        current_section=current_section,
        selector_statuses=selector_statuses,
        new_subsection_form=new_subsection_form, desc_form=desc_form,
        owners_form=owners_form, is_owner=is_owner,
        new_rule_form=new_rule_form, rule_order_form=rule_order_form
    )


def aliases(path):
    storage = flask.current_app.storage
    section, current_section = _get_section(path, max_depth=0)
    if section['stype'] != 'sandbox_resource':
        flask.abort(404)
    rtype = section['stype_options']['resource_type']
    releases = storage.get_volatiles('sandbox_releases', [rtype],
                                     full=True)[rtype]['value'] or []
    descr_by_rid = {rr['resource_id']: rr['description']
                    for rr in releases}
    all_owners = storage.get_all_owners(path)
    is_owner = storage.is_user_in_userlist(
        flask.g.username, flask.g.usergroups, all_owners
    )
    is_editable = (section['revision'] == current_section['revision'] and
                   is_owner)
    if flask.request.method == 'POST':
        formlists = dict(flask.request.form.lists())
        allvals = {}
        for key in 'id name resource'.split():
            allvals[key] = formlists.get(key, [])
        nforms = max(len(v) for v in allvals.values())
        for key in allvals:
            allvals[key].extend([''] * (nforms - len(allvals[key])))
        datadicts = []
        for i in range(nforms):
            md = MultiDict({key: allvals[key][i] for key in allvals})
            md['csrf_token'] = flask.request.form.get('csrf_token')
            datadicts.append(md)
        alias_forms = [forms.AliasForm(releases, formdata=datadict)
                       for datadict in datadicts]
        if all([form.validate_on_submit() for form in alias_forms]):
            aliases = [dict(f.data, resource=int(f.data['resource']),
                            description=descr_by_rid[int(f.data['resource'])])
                       for f in alias_forms]
            storage.save_section_resource_aliases(
                flask.g.username, flask.g.usergroups, section['path'],
                old_revision=section['revision'], aliases=aliases
            )
            flask.flash(gettext('Aliases successfully saved'), 'success')
            return flask.redirect(flask.url_for('section', path=path))
    else:
        alias_forms = [
            forms.AliasForm(releases,
                            data={'id': dd['id'], 'name': dd['name'],
                                  'resource': str(dd['resource_id'])})
            for dd in section['stype_options'].get('aliases', [])
        ]

    all_rules = []
    for rule in section['rules']:
        all_rules.append(rule)
        for subrule in rule['subrules']:
            all_rules.append(subrule)

    for form in alias_forms:
        form.used_in_rules = []
        alias_id = form.data.get('id')
        if form.resource.data and form.resource.data.isdigit():
            form.resource.resource_description = descr_by_rid.get(
                int(form.resource.data)
            )
        if not alias_id:
            continue
        for rule in all_rules:
            if rule['config_source']['rtype'] != 'by_alias':
                continue
            if rule['config_source']['alias_id'] == alias_id:
                form.used_in_rules.append(rule['name'])

    return flask.render_template(
        'aliases.html', section=section,
        current_section=current_section,
        is_owner=is_owner,
        is_editable=is_editable,
        alias_forms=alias_forms,
        empty_form=forms.AliasForm(releases, formdata=None)
    )


def history(path):
    limit = flask.current_app.config['HISTORY_PAGE_SIZE']
    recursive = bool(flask.request.args.get('recursive'))
    page = flask.request.args.get('page')
    if page and page.isdigit():
        page = int(page)
        offset = page * limit
    else:
        offset = 0
        page = 0
    rule = None
    if not recursive:
        rule = flask.request.args.get('rule')
    storage = flask.current_app.storage
    hist = storage.get_section_history(path, offset=offset, limit=limit + 1,
                                       recursive=recursive, rule=rule)
    if page != 0 and not hist:
        flask.abort(404)
    is_last_page = len(hist) <= limit
    hist = hist[:limit]
    is_first_page = page == 0
    section = {'path': path}
    return flask.render_template(
        'history.html', section=section, history=hist, page=page,
        is_last_page=is_last_page, is_first_page=is_first_page,
        recursive_history=recursive, rule=rule,
    )


def revert_rules(path):
    current_rev = flask.request.form.get('current_rev', '')
    revert_to_rev = flask.request.form.get('revert_to_rev', '')
    if not current_rev.isdigit() or not revert_to_rev.isdigit():
        flask.abort(400)
    flask.current_app.storage.revert_rules(
        flask.g.username, flask.g.usergroups,
        path, int(current_rev), int(revert_to_rev)
    )
    flask.flash(gettext('Rules successfully reverted'), 'success')
    return flask.redirect(flask.url_for('section', path=path))


def delete_empty_section(path):
    old_revision = flask.request.args.get('revision')
    if not old_revision or not old_revision.isdigit():
        flask.abort(400)
    old_revision = int(old_revision)
    flask.current_app.storage.delete_empty_section(
        flask.g.username, flask.g.usergroups, path, old_revision
    )
    parent_path = '' if '.' not in path else path.rsplit('.', 1)[0]
    flask.flash(gettext('Section successfully deleted'), 'success')
    return flask.redirect(flask.url_for('section', path=parent_path))


def rule(path, rulename):
    section, current_section = _get_section(path, max_depth=0)
    action = flask.request.form.get('action')
    storage = flask.current_app.storage
    rules_by_name = dict()
    for rule in section['rules']:
        rules_by_name[rule['name']] = (rule, None)
        for subrule in rule['subrules']:
            rules_by_name[subrule['name']] = (subrule, rule)
    if rulename not in rules_by_name:
        flask.abort(404)
    rule, parent_rule = rules_by_name[rulename]

    selector_volatile = None
    if rule['selector'] is not None:
        selector_volatile = storage.get_volatiles(
            'selector', [rule['selector']]
        )[rule['selector']]
    resource_volatile = None

    edit_rule_form = forms.EditRuleForm(data={
        'desc': rule['desc'],
        'editors': rule['editors'],
        'selector': rule['selector'],
        'htype': (forms.EditRuleForm.HTYPE_ALL
                  if rule['selector'] is None else
                  forms.EditRuleForm.HTYPE_SOME),
        'revision': section['revision'],
        'action': 'edit-rule',
    })
    config_data = {
        'revision': section['revision'],
        'action': 'edit-config'
    }

    new_subrule_data = {'revision': section['revision'], 'action': 'new-rule'}
    if section['stype'] == 'yaml':
        new_subrule_form = forms.NewYamlRuleForm(data=new_subrule_data)
    else:
        new_subrule_form = forms.NewSandboxResourceRuleForm(
            resource_type=section['stype_options']['resource_type'],
            aliases=section['stype_options'].get('aliases', []),
            released_resources=_get_released_resources(section),
            data=new_subrule_data
        )
    rule_order_form = None
    if rule.get('subrules'):
        rule_order_form = forms.RuleOrderForm(data={
            'revision': section['revision'],
            'action': 'reorder-rules',
            'order': range(len(rule['subrules'])),
        })

    if section['stype'] == 'yaml':
        config_form = forms.YamlRuleConfigForm(
            config_source=rule['config_source'], data=config_data,
        )
    else:
        config_form = forms.SandboxResourceRuleConfigForm(
            resource_type=section['stype_options']['resource_type'],
            aliases=section['stype_options'].get('aliases', []),
            released_resources=_get_released_resources(section),
            config_source=rule['config_source'], data=config_data
        )
        rn = rule['config']['resource_id']
        resource_volatile = storage.get_volatiles('sandbox_resource', [rn])[rn]

    delete_rule_form = forms.DeleteRuleForm(data={
        'revision': section['revision'],
        'action': 'delete-rule',
    })

    if action == 'edit-rule' and edit_rule_form.validate_on_submit():
        data = edit_rule_form.data
        selector = (data['selector']
                    if data['htype'] == forms.EditRuleForm.HTYPE_SOME
                    else None)
        storage.edit_rule(
            flask.g.username, flask.g.usergroups,
            path, old_revision=data['revision'],
            rulename=rulename, desc=data['desc'], editors=data['editors'],
            selector=selector
        )
        flask.flash(gettext('Rule changes saved'), 'success')
        return flask.redirect(flask.url_for('rule', path=path,
                                            rulename=rulename))

    if action == 'new-rule' and new_subrule_form.validate_on_submit():
        data = new_subrule_form.data
        selector = (data['selector']
                    if data['htype'] == forms.EditRuleForm.HTYPE_SOME
                    else None)
        try:
            storage.create_rule(
                flask.g.username, flask.g.usergroups,
                path, old_revision=data['revision'],
                parent_rule=rule['name'], rulename=data['name'],
                desc=data['desc'], editors=data['editors'], selector=selector,
                config=new_subrule_form.get_config(),
                config_source=new_subrule_form.get_config_source(),
            )
        except errors.NotUnique:
            new_subrule_form['name'].errors.append(
                gettext('Rule name is not unique.')
            )
        else:
            flask.flash(gettext('New subrule created'), 'success')
            return flask.redirect(flask.url_for('rule', path=path,
                                                rulename=data['name']))

    if action == 'reorder-rules' and rule_order_form.validate_on_submit():
        old_revision = rule_order_form.data['revision']
        storage.reorder_rules(flask.g.username, flask.g.usergroups,
                              path, old_revision, parent_rule=rule['name'],
                              new_order=rule_order_form.data['order'])
        flask.flash(gettext('New subrules order saved'), 'success')
        return flask.redirect(flask.url_for('rule', path=path,
                                            rulename=rule['name']))

    if action == 'edit-config' and config_form.validate_on_submit():
        old_revision = config_form.data['revision']
        storage.edit_rule_config(flask.g.username, flask.g.usergroups,
                                 path, old_revision,
                                 rulename, config_form.get_config(),
                                 config_form.get_config_source())
        flask.flash(gettext('Rule config changes saved'), 'success')
        retpath = flask.request.args.get('retpath')
        if not retpath or not retpath.startswith('/'):
            retpath = flask.url_for('rule', path=path, rulename=rulename)
        return flask.redirect(retpath)

    elif action == 'delete-rule' and delete_rule_form.validate_on_submit():
        old_revision = delete_rule_form.data['revision']
        storage.delete_rule(flask.g.username, flask.g.usergroups,
                            path, old_revision, rulename)
        if parent_rule is None:
            flask.flash(gettext('Rule deleted'), 'success')
            return flask.redirect(flask.url_for('section', path=path))
        else:
            flask.flash(gettext('Subrule deleted'), 'success')
            return flask.redirect(flask.url_for('rule', path=path,
                                                rulename=parent_rule['name']))

    is_owner = False
    if section['revision'] == current_section['revision']:
        is_owner = storage.is_owner(flask.g.username, flask.g.usergroups, path)
        if parent_rule and storage.is_user_in_userlist(flask.g.username,
                                                       flask.g.usergroups,
                                                       parent_rule['editors']):
            is_owner = True
    is_editor = storage.is_user_in_userlist(flask.g.username,
                                            flask.g.usergroups,
                                            rule['editors'])

    rule_exists_in_current_revision = any(
        rulename == rule['name'] or any(rulename == subrule['name']
                                        for subrule in rule['subrules'])
        for rule in current_section['rules']
    )

    selector_statuses = None
    if parent_rule is None:
        all_selectors = set(subrule['selector'] for subrule in rule['subrules']
                            if subrule['selector'] is not None)
        selector_statuses = storage.get_volatiles('selector', all_selectors)

    return flask.render_template(
        'rule.html', section=section, rule=rule,
        selector_volatile=selector_volatile,
        resource_volatile=resource_volatile,
        current_section=current_section,
        rule_exists_in_current_revision=rule_exists_in_current_revision,
        is_owner=is_owner, is_editor=is_editor,
        edit_rule_form=edit_rule_form,
        config_form=config_form,
        delete_rule_form=delete_rule_form,
        rule_order_form=rule_order_form,
        new_subrule_form=new_subrule_form,
        parent_rule=parent_rule,
        selector_statuses=selector_statuses,
    )


def force_volatile_update(vtype, key):
    storage = flask.current_app.storage
    vrec = storage.get_volatile(vtype, key)
    timestamp = storage.force_volatile_update(vtype, key)
    poll_url = flask.url_for('get_updated_volatile', vtype=vtype,
                             key=vrec['raw_key'], timestamp=timestamp,
                             _external=True)
    value_url = flask.url_for('raw_volatile_status', vtype=vtype,
                              key=vrec['raw_key'], _external=True)
    return flask.jsonify({'poll_url': poll_url, 'value_url': value_url})


def get_updated_volatile(vtype, key, timestamp):
    storage = flask.current_app.storage
    vrec = storage.get_updated_volatile(vtype, key, timestamp)
    if vrec is None:
        flask.abort(404)
    return ''


def volatile_status(vtype, key):
    storage = flask.current_app.storage
    vrec = storage.get_volatile(vtype, key)
    templates = {
        'selector': 'selector_status.html',
        'section': 'section_status.html',
        'sandbox_releases': 'releases_status.html',
        'sandbox_resource': 'resource_status.html',
    }
    return flask.render_template(templates[vrec['vtype']], volatile=vrec)


def raw_volatile_status(vtype, key):
    storage = flask.current_app.storage
    vrec = storage.get_volatile(vtype, key, full=True)
    return flask.jsonify(vrec)


def show_config():
    hostname = flask.request.args.get('hostname')
    if not hostname:
        flask.abort(404)
    url = '{}/v2/hosts/{}'.format(
        flask.current_app.config['GENISYS_API_URI'], hostname
    )
    resp = requests.get(url, {'fmt': 'msgpack'}, timeout=10)
    resp.raise_for_status()
    config = msgpack.loads(resp.content, encoding='utf8')
    return flask.render_template('host_config.html', config=config,
                                 hostname=hostname,
                                 raw_config_url=url + '?fmt=yaml')


def get_matching_rules(path):
    hostname = flask.request.form.get('hostname')
    if not hostname:
        flask.abort(404)
    section, _ = _get_section(path, max_depth=0)
    selector_statuses = _get_all_selector_statuses(
        section, flask.current_app.storage, full=True
    )
    matching_rule_names = []
    warning_rule_names = []
    for rule in section['rules']:
        if rule['selector'] is None:
            matching_rule_names.append(rule['name'])
        else:
            status = selector_statuses[rule['selector']]
            if status['value'] is None:
                warning_rule_names.append(rule['name'])
                for subrule in rule['subrules']:
                    warning_rule_names.append(subrule['name'])
                continue
            if hostname in status['value']:
                matching_rule_names.append(rule['name'])
            else:
                continue
        for subrule in rule['subrules']:
            if subrule['selector'] is None:
                matching_rule_names.append(subrule['name'])
                continue
            status = selector_statuses[subrule['selector']]
            if status['value'] is None:
                warning_rule_names.append(subrule['name'])
                continue
            if hostname in status['value']:
                matching_rule_names.append(subrule['name'])
    return flask.jsonify(
        matching_rule_names=matching_rule_names,
        warning_rule_names=warning_rule_names,
    )


def ping():
    status = 200 if flask.current_app.storage.is_alive() else 503
    return flask.current_app.response_class(status=status)


def group(groupname):
    # https://staff-api.yandex-team.ru/v3/persons?_doc=1
    cfg = flask.current_app.config
    params = {'_one': '1',
              '_fields': 'type,parent.service.id,service.id',
              'url': groupname}
    resp = requests.get(cfg['STAFF_URI'] + '/groups',
                        params=params,
                        headers=cfg['STAFF_HEADERS'],
                        timeout=cfg['STAFF_TIMEOUT'],
                        allow_redirects=False)
    resp.raise_for_status()
    resp = resp.json()
    grouptype = resp['type']
    if grouptype == 'department':
        url = 'https://staff.yandex-team.ru/departments/{}'.format(groupname)
    elif grouptype == 'service' \
            and 'service' in resp \
            and 'id' in resp['service']:
        url = 'https://abc.yandex-team.ru/services/{}/'.format(
            resp['service']['id']
        )
    elif grouptype == 'servicerole' \
            and 'parent' in resp \
            and 'service' in resp['parent'] \
            and 'id' in resp['parent']['service']:
        url = 'https://abc.yandex-team.ru/services/{}/'.format(
            resp['parent']['service']['id']
        )
    else:
        # fallback to (pretty useless) general group page
        url = 'https://staff.yandex-team.ru/groups/{}'.format(groupname)
    return flask.redirect(url)
