package cpuinfo

// https://github.com/shirou/gopsutil/blob/master/cpu/cpu_linux.go
// no regrets

import (
	"io/ioutil"
	"regexp"
	"strconv"
	"strings"
)

const (
	CPUInfoPath = "/proc/cpuinfo"
)

type CPU struct {
	ID         int64
	VendorID   string
	Model      int64
	Family     int64
	ModelName  string
	Flags      []string
	Cores      int64
	MHz        float64
	CacheSize  int64
	PhysicalID int64
	CoreID     int64
}

type CPUInfo struct {
	CPUs []CPU
}

func (c *CPUInfo) NumCPU() int {
	return len(c.CPUs)
}

func (c *CPUInfo) NumPhysCore() int {
	core := make(map[string]bool)

	for _, p := range c.CPUs {
		pid := p.PhysicalID
		cid := p.CoreID

		if pid == -1 {
			return c.NumCPU()
		} else {
			// to avoid fmt import
			key := strconv.FormatInt(int64(pid), 10) + ":" + strconv.FormatInt(int64(cid), 10)
			core[key] = true
		}
	}

	return len(core)
}

func (c *CPUInfo) NumSockets() int {
	pcpu := make(map[string]bool)

	for _, p := range c.CPUs {
		pid := p.PhysicalID

		if pid == -1 {
			return c.NumCPU()
		} else {
			// to avoid fmt import
			key := strconv.FormatInt(int64(pid), 10)
			pcpu[key] = true
		}
	}

	return len(pcpu)
}

var cpuinfoRegExp = regexp.MustCompile(`([^:]*?)\s*:\s*(.*)$`)

func ReadCPUInfo(path string) (*CPUInfo, error) {
	b, err := ioutil.ReadFile(path)
	if err != nil {
		return nil, err
	}

	content := string(b)
	lines := strings.Split(content, "\n")

	var cpuinfo = CPUInfo{}
	var cpu = &CPU{CoreID: -1, PhysicalID: -1}

	for i, line := range lines {
		var key string
		var value string

		if len(line) == 0 && i != len(lines)-1 {
			// end of processor
			cpuinfo.CPUs = append(cpuinfo.CPUs, *cpu)
			cpu = &CPU{}
			continue
		} else if i == len(lines)-1 {
			continue
		}

		submatches := cpuinfoRegExp.FindStringSubmatch(line)
		key = submatches[1]
		value = submatches[2]

		switch key {
		case "processor":
			cpu.ID, _ = strconv.ParseInt(value, 10, 64)
		case "vendor_id":
			cpu.VendorID = value
		case "model":
			cpu.Model, _ = strconv.ParseInt(value, 10, 64)
		case "cpu family":
			cpu.Family, _ = strconv.ParseInt(value, 10, 64)
		case "model name":
			cpu.ModelName = value
		case "flags":
			cpu.Flags = strings.Fields(value)
		case "cpu cores":
			cpu.Cores, _ = strconv.ParseInt(value, 10, 64)
		case "cpu MHz":
			cpu.MHz, _ = strconv.ParseFloat(value, 64)
		case "cache size":
			cpu.CacheSize, _ = strconv.ParseInt(value[:strings.IndexAny(value, " \t\n")], 10, 64)
			if strings.HasSuffix(line, "MB") {
				cpu.CacheSize *= 1024
			}
		case "physical id":
			cpu.PhysicalID, _ = strconv.ParseInt(value, 10, 64)
		case "core id":
			cpu.CoreID, _ = strconv.ParseInt(value, 10, 64)
		}
	}
	return &cpuinfo, nil
}
