package ipc

import (
	"errors"

	"a.yandex-team.ru/infra/gopcm/pkg/pcm"
)

const (
	theorMax = 4.0
)

type IPC struct {
	pcm          *pcm.Pcm
	beforeStates []*pcm.MsrCounterState
	afterStates  []*pcm.MsrCounterState
	val          Values
	numCores     int
}

func programIPC(p *pcm.Pcm) error {
	for i := 0; i < p.NumCores(); i++ {
		err := p.ProgramDefCoreCounters(i)
		if err != nil {
			return err
		}
	}
	return nil
}

func Init(p *pcm.Pcm) (*IPC, error) {

	numCores := p.NumCores()

	err := programIPC(p)
	if err != nil {
		return nil, err
	}

	before := make([]*pcm.MsrCounterState, p.NumCores())
	after := make([]*pcm.MsrCounterState, p.NumCores())
	for i := 0; i < numCores; i++ {
		before[i] = new(pcm.MsrCounterState)
		after[i] = new(pcm.MsrCounterState)
	}

	for i := 0; i < numCores; i++ {
		err = p.ReadMsrCounters(i, before[i])
	}
	res := &IPC{
		pcm:          p,
		beforeStates: before,
		afterStates:  after,
		val:          make(Values, p.NumCores()),
		numCores:     numCores,
	}

	return res, err
}

func (ipc *IPC) Update(reprogram bool) error {
	var resErr string
	for i := 0; i < ipc.numCores; i++ {
		err := ipc.pcm.ReadMsrCounters(i, ipc.afterStates[i])
		if err != nil {
			resErr += err.Error()
		}

	}

	for i := 0; i < ipc.pcm.NumCores(); i++ {
		val := ipc.pcm.GetIPC(ipc.beforeStates[i], ipc.afterStates[i])
		// running 'perf stat -a' increases IPC values
		// to some strane values like "1.2238462989022365e+09" for a short period of time
		// need to understand why
		if val > theorMax {
			ipc.val[i] = theorMax
		} else {
			ipc.val[i] = val
		}
	}

	if reprogram {
		if ipc.val.isAllZeros() {
			err := programIPC(ipc.pcm)
			if err != nil {
				return err
			}
		}
	}

	ipc.beforeStates, ipc.afterStates = ipc.afterStates, ipc.beforeStates
	if len(resErr) != 0 {
		return errors.New(resErr)
	}
	return nil
}

func (ipc *IPC) Values() Values {
	return ipc.val
}
