package mcfg

import (
	"errors"
	"fmt"
	"io/ioutil"
)

type MCFGContent []byte
type MCFGHeaderContent []byte
type MCFGRecordsContent []byte

func readContent() (MCFGContent, error) {
	buf := MCFGContent{}
	for _, p := range MCFGPaths {
		b, err := ioutil.ReadFile(p)
		if err != nil {
			continue
		}
		buf = MCFGContent(b)
		break
	}
	if len(buf) == 0 {
		msg := fmt.Sprintf("Could not read any of %v", MCFGPaths)
		return nil, errors.New(msg)
	}
	return buf, nil
}

func (mc MCFGContent) check() error {
	lmc := len(mc)
	minSize := MCFGHeaderSize + MCFGRecordSize
	if lmc < minSize {
		msg := fmt.Sprintf("Strange size of MCFG content: %d, want at least", minSize)
		return errors.New(msg)
	}
	return nil
}

func (mc MCFGContent) headerContent() (MCFGHeaderContent, error) {
	hc := MCFGHeaderContent(mc[:MCFGHeaderSize])
	return hc, nil
}

func (mc MCFGContent) recordsContent() (MCFGRecordsContent, error) {
	lrc := len(mc) - MCFGHeaderSize
	if lrc%MCFGRecordSize != 0 || lrc < 16 {
		msg := fmt.Sprintf("Length of records segment is: %d", lrc)
		return nil, errors.New(msg)
	}
	rc := MCFGRecordsContent(mc[MCFGHeaderSize:])
	return rc, nil
}

func (mr MCFGRecordsContent) numberOfRecords() int {
	return len(mr) / MCFGRecordSize
}
