package pci

import (
	"bytes"
	"errors"
	"fmt"
	"os"

	"a.yandex-team.ru/infra/gopcm/internal/endian"
)

type PciHandle struct {
	fd       *os.File
	bus      uint8
	device   uint32
	function uint32
}

func Init(groupNr uint16, bus uint8, device, function uint32) (*PciHandle, error) {
	handle, err := openHandle(groupNr, bus, device, function)
	if err != nil {
		return nil, err
	}
	p := &PciHandle{
		fd:       handle,
		bus:      bus,
		device:   device,
		function: function,
	}
	return p, nil
}

func openHandle(groupNr uint16, bus uint8, device, function uint32) (*os.File, error) {
	var path string
	basePath := "/proc/bus/pci/"
	if groupNr > 0 {
		basePath = fmt.Sprintf("%s%04x:", basePath, groupNr)
	}
	path = fmt.Sprintf("%s%02x/%02x.%x", basePath, bus, device, function)

	handle, err := os.OpenFile(path, os.O_RDWR, 0600)
	if err != nil {
		return nil, err
	}
	return handle, nil
}

func (p *PciHandle) Close() error {
	return p.fd.Close()
}

func (p *PciHandle) Read32(offset uint32) (uint32, error) {
	native := endian.Endian()
	b := make([]byte, 4)
	buf := bytes.NewBuffer(b)
	_, err := p.fd.ReadAt(b, int64(offset))
	if err != nil {
		return 0, err
	}
	return native.Uint32(buf.Next(4)), nil
}

func (p *PciHandle) Write32(value, offset uint32) error {
	native := endian.Endian()
	b := make([]byte, 4)
	native.PutUint32(b, value)
	_, err := p.fd.WriteAt(b, int64(offset))
	if err != nil {
		return err
	}
	return nil
}

func (p *PciHandle) Read64(offset uint32) (uint64, error) {
	native := endian.Endian()
	b := make([]byte, 8)
	buf := bytes.NewBuffer(b)
	_, err := p.fd.ReadAt(b, int64(offset))
	if err != nil {
		return 0, err
	}
	return native.Uint64(buf.Next(8)), nil
}

func (p *PciHandle) Write64(value uint64, offset uint32) error {
	return errors.New("unsupported write64")
}

func exists(groupNr uint16, bus uint8, device, function uint32) bool {
	handle, err := openHandle(groupNr, bus, device, function)
	if err != nil {
		return false
	}
	_ = handle.Close()
	return true
}

func CreateIntelPerfMonDevice(groupNr uint16, bus uint8, device, function uint32) (*PciHandle, error) {
	if exists(groupNr, bus, device, function) {
		h, err := Init(groupNr, bus, device, function)
		if err != nil {
			return nil, err
		}
		return h, nil
	}
	return nil, nil
}
