import sys
CYGWIN = (sys.platform == 'cygwin')

if not CYGWIN:
    if __import__('os').uname().version.startswith('FreeBSD 7'):
        __import__('os').supports_dir_fd.clear()

if __file__ == 'frozen' or CYGWIN:
    frozen = True
else:
    frozen = False
    import dns.resolver

import argparse
import itertools
import json
import os
import pwd
import random
import re
import shlex
import shutil
import signal
import socket
import stat
import struct
import subprocess as subproc
import syslog
import time
import urllib.parse
import urllib.request
import yaml

from collections import OrderedDict
from py.path import local as Path  # noqa

from .version import LooseVersion


SKY_GET_TIMEOUT = 15 * 60
RSYNC_GET_TIMEOUT = 15 * 60
HTTP_GET_TIMEOUT = 15 * 60
SKYNETBIN_RUN_TIMEOUT = 20 * 60
SKYNETBIN_RESHARE_TIMEOUT = 2 * 60
SKYDIST_DIR = Path('/var/tmp/skydist')
STORED_CONTEXT_FILE = SKYDIST_DIR.join('.data')

if CYGWIN:
    FREEBSD = False
    BSD = True
else:
    FREEBSD = os.uname().version.startswith('FreeBSD')
    BSD = FREEBSD or os.uname().version.startswith('Darwin')

MAGIC = b'\xf2K^j\xb2\xc7\xf6\x92u\x1b\xed{\x10\xc1S\xcefg\xe2\xcb'

try:
    syslog.openlog(ident='gosky')
except:
    pass


def platform():
    """
    The method actually determines current platform name in the same manner as `skynet.bin` executable does,
    but with special code related to Linux compatibility layer running on FreeBSD.
    :return: Platform name (OS name plus achitecture name).
    """
    if FREEBSD:
        host = 'freebsd' + os.uname().version[8]  # e.g. freebsd8
        arch = subproc.check_output(['uname', '-m']).decode('ascii').strip()
    else:
        host, arch = os.uname()[0], os.uname()[4]
        host = host.lower()

    if arch in ('i686', 'i586', 'i486', 'i386'):
        arch = 'x86'
    elif arch in ('amd64', 'x86_64'):
        arch = 'amd64'
    else:
        arch = arch

    return '%s_%s' % (host, arch)


def check_nice_revertability():
    """
        Get lowest possible nice, which we will be able
        to upnice back to 0.

        In FreeBSD jails we cant upnice even with root, so
        this function will return 0
    """
    r, w = os.pipe()
    pid = os.fork()

    if pid == 0:
        try:
            current_nice = os.nice(0)
            current_nice2 = os.nice(19 - current_nice)
            current_nice3 = os.nice(0 - current_nice2)
            assert current_nice3 == 0
            os.write(w, struct.pack('!b', (19 - current_nice)))
        except:
            os.write(w, struct.pack('!b', 0))
        finally:
            raise SystemExit(0)

    os.waitpid(pid, 0)
    result = os.read(r, 1)
    os.close(r)
    os.close(w)
    return struct.unpack('!b', result)[0]


def log(msg, *args, **kwargs):
    stream = kwargs.pop('stream', sys.stdout)
    syslog_level = syslog.LOG_ERR if stream is sys.stderr else syslog.LOG_INFO
    syslog.syslog(syslog_level, msg % args)
    stream.write((msg + '\n') % args)


def info(msg, *args):
    return log('>>> %s' % (msg, ), *args, stream=sys.stdout)


def warn(msg, *args):
    return log('WRN %s' % (msg, ), *args, stream=sys.stderr)


def error(code, msg, *args):
    log('ERR %s' % (msg, ), *args, stream=sys.stdout)
    raise SystemExit(code)


def parse_svn_url(url):
    scheme, url = url.split('://', 1)
    parts = url.split('/')[1:]

    if parts[:3] == ['arc', 'tags', 'skynet']:
        tag = parts[-1]
        if tag.startswith('stable-'):
            return tag[len('stable-'):].replace('-', '.')
        elif re.match(r'\d+\.\d+\.\d+', tag):
            return tag

    if parts[:3] == ['arc', 'trunk', 'arcadia']:
        return 'trunk'

    if parts[:3] == ['arc', 'branches', 'skynet']:
        return 'branch-%s' % (parts[-1], )

    return '_'.join(parts)


def is_we_fastbonized():
    host = socket.getfqdn()

    # First try if we have hostname.fb.yandex.ru
    try:
        if host.endswith('.yandex.ru'):
            test_name = host[:-len('.yandex.ru')] + '.fb.yandex.ru'
        else:
            test_name = host + '.fb.yandex.ru'

        ip = socket.getaddrinfo(test_name, 0)[0][4][0]
        if socket.gethostbyaddr(ip)[0] != 'any.yandex.ru':
            info('Fastbone mode: %s' % (test_name, ))
            return True
        else:
            warn('%s -> %s -> any.yandex.ru' % (test_name, ip))
    except:
        pass

    # Next try if we have fb-hostname
    try:
        test_name = 'fb-' + host
        ip = socket.getaddrinfo(test_name, 0)[0][4][0]
        if socket.gethostbyaddr(ip)[0] != 'any.yandex.ru':
            info('Fastbone mode: %s' % (test_name, ))
            return True
        else:
            warn('%s -> %s -> any.yandex.ru' % (test_name, ip))
    except:
        pass

    warn('Backbone mode!')
    return False


def get_stored_context(hostname):
    path = STORED_CONTEXT_FILE

    infodict = dict.fromkeys(['last_modified', 'platform', 'config_hash', 'hostname'])

    try:
        if path.check(file=1, exists=1):
            lines = path.readlines()
            for line in lines:
                if any(line.startswith(prefix + ': ') for prefix in infodict.keys()):
                    key, value = map(str.strip, line.split(':', 1))
                    if value == 'None':
                        value = None
                    infodict[key] = value
                    if all(v is not None for v in infodict.values()):
                        break

        if infodict['platform'] is not None and infodict['platform'] != platform():
            warn('Platform mismatch %r -> %r. Forcing update.', infodict['platform'], platform())

            infodict['last_modified'] = None
            infodict['config_hash'] = None

        elif infodict['hostname'] is not None and infodict['hostname'] != hostname:
            warn('Hostname mismatch %r -> %r. Forcing update.', infodict['hostname'], hostname)

            infodict['last_modified'] = None
            infodict['config_hash'] = None

        info('Loaded context: %s', '; '.join(sorted('%s=%s' % (k[0], k[1]) for k in infodict.items())))
    except Exception as ex:
        info('Unable to load context: %s', ex)

    return infodict


def save_stored_context(rawinfo):
    path = STORED_CONTEXT_FILE

    infodict = dict.fromkeys(['last_modified', 'platform', 'config_hash', 'hostname'])

    for key, value in infodict.items():
        infodict[key] = rawinfo.get(key)

    info('Storing context: %s', '; '.join(sorted('%s=%s' % (k[0], k[1]) for k in infodict.items())))

    path.dirpath().ensure(dir=1)
    with path.open('w') as fp:
        for key, value in infodict.items():
            fp.write('%s: %s\n' % (key, value))


def get_freezed_release():
    fp = Path('/Berkanavt/skynet/etc/release.conf')
    if not fp.check(exists=1) or not fp.check(file=1):
        return

    try:
        data = yaml.load(fp.open(mode='rb'))
    except Exception as ex:
        warn('Unable to load skynet version from frozen file: %s', ex)
        return None

    urls = data['urls']

    urls_ordered = OrderedDict()

    # Parse http and rsync urls
    for key, urllist in urls.items():
        if key in ('rsync', 'http'):
            urls[key] = tuple(urllib.parse.urlparse(url) for url in urllist)

    for key in 'skynet', 'rsync', 'http':
        if key in urls:
            urls_ordered[key] = urls[key]

    return {
        'svn_url': data['svn_url'],
        'urls': urls_ordered,
        'md5': data['md5'],
        'version': data['version'],
        'advertizer': str(fp),
    }


def get_hostname():
    info('Detect proper hostname')
    if 'GOSKY_HOSTNAME' in os.environ:
        info('  using GOSKY_HOSTNAME environment variable => %s', os.environ['GOSKY_HOSTNAME'])
        return os.environ['GOSKY_HOSTNAME']

    hostname = socket.gethostname()
    info('  gethostname => %s', hostname)
    info('  getaddrinfo:')
    names = set()
    seen_ips = set()

    try:
        addrinfo = socket.getaddrinfo(hostname, 0, 0, socket.SOCK_STREAM, 0)
    except socket.gaierror as ex:
        if ex.errno in (
            getattr(socket, 'EAI_NONAME', None),
            getattr(socket, 'EAI_NODATA', None),
        ):
            info('    not found (%s)', str(ex))
            info('  using: %s (unable to get ips)', hostname)
            return hostname
        raise
    else:
        for ipinfo in addrinfo:
            if ipinfo[4][0] in seen_ips:
                continue
            seen_ips.add(ipinfo[4][0])

            if ipinfo[0] == socket.AF_INET:
                v6 = False
            else:
                v6 = True

            info('    (%s) %s', 'v4' if not v6 else 'v6', ipinfo[4][0])

            try:
                name = socket.gethostbyaddr(ipinfo[4][0])[0]
                info('      name: %s', name)
                names.add(name)
            except socket.gaierror as ex:
                if ex.errno in (
                    getattr(socket, 'EAI_NONAME', None),
                    getattr(socket, 'EAI_NODATA', None)
                ):
                    info('      name: not found')
                    continue
                else:
                    raise
            except socket.herror as ex:
                if ex.errno == 1:
                    if ex.args[1] == 'Unknown host':
                        continue
                raise

        names = list(names)

        if hostname in names:
            info('  using: %s (hostname in names)', hostname)
            return hostname
        elif len(names) == 1 and names[0].startswith(hostname):
            info('  using: %s (one name)', names[0])
            return names[0]
        else:
            info('  using: %s (>1 fqdn to choose from)', hostname)
            return hostname


def get_advertised_release(hostname, modified_since=None, config_hash=None):
    url = 'http://api.genisys.yandex-team.ru/v2/hosts/%s/skynet.versions?fmt=json&config_hash=%s' % (
        hostname,
        config_hash or ''
    )

    info('Getting %s...' % (url, ))

    try:
        opener = urllib.request.build_opener()
        if modified_since is not None:
            opener.addheaders = [
                ('If-Modified-Since', modified_since),
                ('User-Agent', 'Gosky/2.0')
            ]
        req = opener.open(url, timeout=120)
        last_modified = req.headers.get('Last-Modified', None)
        for header_name, header_value in sorted(req.headers.items()):
            if header_name.startswith('X-Genisys'):
                info('  %s: %s', header_name, header_value)
        data = json.loads(req.read().decode('utf-8'))
    except urllib.error.HTTPError as ex:
        if ex.code == 404:
            error(12, 'Version not specified for this host (%s)', hostname)
        if ex.code == 304:
            info('Configuration date: %s', modified_since)
            info('Configuration was not changed since that, no-op')
            raise SystemExit(0)

        error(12, 'Failed to grab advertised release:\n%s', str(ex))

    config = data['config']

    try:
        svn_url = config['attributes']['svn_url']
        attributes = config['attributes']

        try:
            urls = OrderedDict((
                ('skynet', config['skynet_id']),
                ('http', {100: config['http']['links']}),
                ('rsync', config['rsync']['links']),  # deprioritize rsync
            ))

            for key, value in attributes.items():
                if key.startswith('http_'):
                    if key.endswith('_weight'):
                        continue
                    weight = int(attributes.get(key + '_weight', 5))
                    urls['http'].setdefault(weight, []).append(value)

            for key, value in urls.items():
                if isinstance(value, str):
                    urls[key] = (value, )
                elif isinstance(value, list):
                    random.shuffle(value)
                    urls[key] = tuple(value)
                elif isinstance(value, dict):
                    sorted_by_weight_and_shuffled = []
                    for weight in sorted(value, reverse=True):
                        urls_copied = value[weight][:]
                        random.shuffle(urls_copied)
                        sorted_by_weight_and_shuffled.extend(urls_copied)
                    urls[key] = tuple(sorted_by_weight_and_shuffled)

            md5 = config['md5']
        except KeyError as ex:
            raise KeyError('config:%s' % (ex.args[0], ))

    except KeyError as ex:
        error(12, 'Fail to analyze genisys packet: missing %s key', ex.args[0])
        raise SystemExit(1)

    version = config.get('attributes', {}).get('version', None)
    if version is None:
        version = parse_svn_url(svn_url)

    # Parse http and rsync urls
    for key, urllist in urls.items():
        if key in ('rsync', 'http'):
            urls[key] = tuple(urllib.parse.urlparse(url) for url in urllist)

    return {
        'svn_url': svn_url,
        'version': version,
        'last_modified': last_modified,
        'config_hash': data.get('config_hash'),
        'urls': urls,
        'md5': md5,
        'advertizer': 'genisys',
    }


def chown_chmod_if_needed(path):
    if path.stat().uid != 0:
        path.chown('root', -1)

    if not path.stat().mode & stat.S_IXUSR:
        path.chmod(0o755)


def system(cmd, timeout):
    info('Running `%s`... (timeout = %d mins)', subproc.list2cmdline(cmd), timeout / 60)
    proc = subproc.Popen(cmd)
    if CYGWIN:
        try:
            proc.wait()
            return proc.returncode
        except:
            warn('timed out, killing with SIGKILL...')
            proc.kill()
            time.sleep(1)
            if proc.poll() is None:
                proc.kill()
            return None
    else:
        try:
            proc.wait(timeout=timeout)
            return proc.returncode
        except subproc.TimeoutExpired:
            warn('timed out, killing with SIGKILL...')
            proc.kill()
            time.sleep(1)
            if proc.poll() is None:
                proc.kill()
            return None


def download_skybin_via_skynet(skybin, resid, net):
    if net == 'backbone':
        speedlim = ['--max-dl-speed', '8mbps', '--max-ul-speed', '8mbps']
    else:
        speedlim = []

    cmd = [
        '/skynet/tools/sky',
        'get', '-w', '-p',
        '-P', 'High',
        '-N', 'Backbone' if net == 'backbone' else 'Fastbone',
        '-d', skybin.dirpath().strpath,
    ] + speedlim + [
        resid,
    ]

    rc = system(cmd, SKY_GET_TIMEOUT)
    if rc == 0:
        chown_chmod_if_needed(skybin)
        return skybin

    if rc:
        warn('')
        warn('sky get failed with exit code %d', rc)
    return None


def download_skybin_via_rsync(skybin, url, net):
    proc = subproc.Popen(['rsync', '--help'], stdout=subproc.PIPE, stderr=subproc.PIPE)
    stdout, _ = proc.communicate()
    proc.wait()

    append_verify = []
    contimeout = []
    if proc.returncode == 0:
        if b'--append-verify' in stdout:
            append_verify = ['--append-verify']
        if b'--contimeout' in stdout:
            contimeout = ['--contimeout=10']

    skybin_tmp = skybin.dirpath().join(skybin.basename + '.rsync-download')
    cmd = [
        'rsync',
        '--append'
    ] + append_verify + [
        '--checksum',
        '--inplace',
        '--copy-links',
    ] + contimeout + [
        '--timeout=60',
        '--temp-dir=/var/tmp',
        '--progress',
        '--bwlimit=1000',  # 1mb/s
        url, skybin_tmp.strpath
    ]

    rc = system(cmd, RSYNC_GET_TIMEOUT)
    if rc == 0:
        if skybin.check(exists=1):
            skybin.remove()
        skybin_tmp.move(skybin)
        chown_chmod_if_needed(skybin)
        return skybin
    if rc:
        warn('')
        warn('%r failed with exit code %d', cmd[0], rc)
    return None


def download_skybin_via_http(skybin, url, net):
    skybin_tmp = skybin.dirpath().join(skybin.basename + '.http-download')
    if 'storage-int.mds.yandex.net/get-skynet' in url:
        auth = ['--header', 'Authorization: Basic c2t5bmV0OjNhMjEzODMzYjJlMzY1ZDZiNzQwYjNlMTY0NzVkMWU1']
    else:
        auth = []

    cmd = [
        'wget',
        '-c', '-O', skybin_tmp.strpath,
        '--progress=dot:mega',
        '--connect-timeout=10',
        '--read-timeout=30',
        '--retry-connrefused',
        '--tries=5',
        '--limit-rate=1000000',  # 1mb/s
    ] + auth + [
        url
    ]

    wget_available = next(itertools.chain(
        (x for x in os.environ["PATH"].split(":") if os.path.exists(os.path.join(x, 'wget'))),
        itertools.repeat(None)
    ))
    if not wget_available:
        info("'wget' tool is not available. Using 'fetch' instead.")
        cmd = ['fetch', '-r', '-o', skybin_tmp.strpath, '-T', '30', '-a', url]

    rc = system(cmd, HTTP_GET_TIMEOUT)
    if rc == 0:
        if skybin.check(exists=1):
            skybin.remove()
        skybin_tmp.move(skybin)
        chown_chmod_if_needed(skybin)
        return skybin
    if rc:
        warn('')
        warn('%r failed with exit code %d', cmd[0], rc)
    return None


def download_skybin(version, urls, md5):
    info('Attemping to download skynet binary...')

    skybin = Path('/var/tmp/skydist').join(version, 'skynet.bin')
    skybin.dirpath().ensure(dir=1)
    chown_chmod_if_needed(skybin.dirpath())
    chown_chmod_if_needed(skybin.dirpath().dirpath())

    # Drop > 5th existent binaries
    old_binaries = []
    for path in skybin.dirpath().dirpath().listdir():
        if path == skybin.dirpath():
            # Simple protection against current skynet binary
            continue

        if path == STORED_CONTEXT_FILE:
            # Protection against context file
            continue

        old_binaries.append((path, LooseVersion(path.basename)))

    old_binaries.sort(key=lambda x: x[1])

    for path, version in old_binaries[:-4]:  # we excluded current binary, so must leave 4 more
        info('dropping "%s"...' % (path, ))
        path.remove()

    # Check for free space before attemping to download
    try:
        skybin_parent_free = shutil.disk_usage(skybin.dirpath().strpath).free
    except AttributeError:
        # Does not work on CYGWIN
        pass
    except OSError as ex:
        if ex.errno == 75:
            # too much blocks
            pass
        else:
            raise
    else:
        if skybin_parent_free < 300 << 20:
            error(
                11,
                'There is not enough free space in "%s": only %dMiB free (300MiB required).',
                skybin.dirpath(), skybin_parent_free >> 20
            )

    if skybin.check(exists=1) or skybin.check(link=1):
        if skybin.check(file=1):
            if skybin.computehash() == md5:
                info('binary already downloaded and md5 looks sane')
                return skybin
        else:
            skybin.remove()

    if is_we_fastbonized():
        possible_networks = ('fastbone', 'fastbone2', 'backbone')
    else:
        possible_networks = ('backbone', )

    possible_tries = 0
    for key, urllist in urls.items():
        for url in urllist:
            if key == 'skynet':
                # skynet transport does not support 'fastbone2' mode
                nets = set(possible_networks)
                nets.discard('fastbone2')
                possible_tries += len(nets)
            else:
                possible_tries += len(possible_networks)

    global_tryout = 0
    for tryout, net in enumerate(possible_networks):
        for key, urllist in urls.items():
            func = globals().get('download_skybin_via_%s' % (key, ), None)
            if func:
                for url in urllist:
                    if isinstance(url, urllib.parse.ParseResult):
                        # Extract pure hostname from url
                        components = list(url)
                        user, host = urllib.parse.splituser(components[1])
                        host, port = urllib.parse.splitport(host)

                        if net == 'fastbone':
                            if host.endswith('.yandex.ru'):
                                host = host[:-len('.yandex.ru')]
                            host = host + '.fb.yandex.ru'
                        elif net == 'fastbone2':
                            host = 'fb-' + host

                        # Now put components with modified hostname back
                        if port is not None:
                            host = '%s:%s' % (host, port)
                        if user is not None:
                            host = '%s@%s' % (user, host)

                        components[1] = host
                        url = urllib.parse.urlunparse(components)

                    elif url.startswith('rbtorrent'):
                        # For rbtorrent urls dont try 'fastbone2' mode
                        if net in ('fastbone2', ):
                            continue
                    try:
                        global_tryout += 1
                        info('-' * 80)
                        info(
                            '[%d/%d] Trying to download with %r from %r',
                            global_tryout, possible_tries, func, url
                        )
                        result = func(skybin, url, net)
                        info('-' * 80)
                    except Exception as ex:
                        info('-' * 80)
                        warn('failed with unhandled exception: %s', ex)
                        result = None

                    if result:
                        return result
            else:
                warn('skip url type "%s": dont know how to download it', key)


def get_active_info():
    old_svn_url = None
    old_svn_revision = None
    old_version = None
    old_state = None

    info = Path('/skynet/.info')

    try:
        if info.check(exists=1, file=1):
            for line in info.readlines():
                if not old_svn_revision and line.startswith('revision: '):
                    old_svn_revision = line[10:].strip()

                if not old_svn_url and line.startswith('url: '):
                    old_svn_url = line[5:].strip()
                    if old_svn_url.endswith('/skynet'):
                        old_svn_url = old_svn_url[:-len('/skynet')]

                if not old_version and line.startswith('version: '):
                    old_version = line[9:].strip()

                if not old_state:
                    if line.startswith('state: '):
                        old_state = line[7:].strip()
                    elif line.startswith('_url: '):
                        old_state = 'forced_no_url'

                if old_svn_url and old_svn_revision and old_version and old_state:
                    break

            if not old_state:
                # For now if where is no state in .info and url is okay (not mangled)
                old_state = 'no_state'  # old ok :)
        else:
            old_state = 'no_info'

    except Exception as ex:
        warn('Fail to read old skynet svn url: %s', ex)

    return old_svn_url, old_svn_revision, old_version, old_state


def run_skybin(ctx, skybin):
    cmd = [
        skybin.strpath,
        '-q', '-b', '--log',
        '--tmpdir', '/var/tmp',
        '/'
    ]
    if ctx['args'].force:
        cmd.insert(-1, '--force-restart')

    if ctx['args'].opts:
        cmd = cmd[:-1] + shlex.split(ctx['args'].opts) + [cmd[-1]]

    try:
        try:
            rc = system(cmd, SKYNETBIN_RUN_TIMEOUT)
        except OSError as ex:
            warn('skynet.bin: text file busy, will retry after 10 seconds')
            if ex.errno == 26:  # text file busy
                time.sleep(10)
            rc = system(cmd, SKYNETBIN_RUN_TIMEOUT)
    except Exception as ex:
        error(11, 'skynet.bin failed to run: %s' % (str(ex), ))
        return 1

    if rc == 17:
        # skybin fails self-check. Remove it, so
        # we will redownload in future.
        if skybin.check(exists=1):
            skybin.remove()

    if rc in (0, 17):
        # If everything finished ok, we should leave
        # only final skynet.bin, not matter that temporarily
        # files were also downloaded.
        #
        # If skybin fails with self-check, we also remove all
        # temporary files, to start next time from the scratch.
        for path in skybin.dirpath().listdir():
            if path != skybin:
                path.remove()

    return rc


def reshare_skybin(skybin):
    cmd = [
        '/skynet/tools/sky',
        'share',
        '-d', skybin.dirpath().strpath,
        skybin.basename
    ]
    return system(cmd, SKYNETBIN_RESHARE_TIMEOUT)


def random_sleep(seconds, sock):
    seconds = random.randint(0, seconds)
    try:
        while seconds > 0:
            now = time.time()
            info('  Sleep for  : %d seconds', seconds)
            sock.settimeout(seconds)
            sock.listen(1)
            peer = sock.accept()[0]
            seconds -= time.time() - now
            peer.settimeout(seconds)
            if peer.recv(len(MAGIC)) == MAGIC:
                info('Lock release requested. Terminating.')
                sock.close()
                os._exit(1)
            seconds -= time.time() - now
            peer.close()
    except socket.timeout:
        pass
    sock.settimeout(None)
    return seconds


def lock_on_port(port, releaseable=False):
    host_name = '127.0.0.1'
    socket_name = '\x00gosky.lock'  # Linux only. Not applicable on Unix.

    newsock = lambda: socket.socket(socket.AF_INET if BSD else socket.AF_UNIX, socket.SOCK_STREAM)
    sock = newsock()
    sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    bindon = (host_name, port) if BSD else socket_name
    try:
        sock.bind(bindon)
    except OSError:
        if releaseable:
            info('Attempt to release the lock.')
            try:
                sock = newsock()
                sock.connect(bindon)
                sock.send(MAGIC)
                sock.settimeout(1)
                if not sock.recv(1):
                    return lock_on_port(port, releaseable=False)
            except socket.error:
                pass
        error(
            11,
            'lock already held on %r, exiting.',
            '%s:%d' % (host_name, args.lock_port) if BSD else '@%s' % socket_name[1:]
        )

    if isinstance(bindon, (list, tuple)):
        bindon_string = '%s:%d' % (bindon[0], bindon[1])
    else:
        bindon_string = socket_name.replace('\x00', '@')

    return bindon_string, sock


if __name__ == '__main__':
    """
    Exit codes:
        1..10 -- skynet.bin error
        11    -- no skynet user, unable to get skynet.bin, unable to run skynet.bin, no free space, etc.
        12    -- gosky lock already held, genisys errors
    """
    if not frozen:
        dns.resolver.override_system_resolver()

    # We will 100% need skynet user, so check it now
    try:
        pwd.getpwnam('skynet')
    except KeyError:
        error(11, 'ERROR: skynet user was not found in system.')

    parser = argparse.ArgumentParser()
    parser.add_argument('-q', '--quiet', action='store_true', help='be quiet')
    parser.add_argument('-l', '--lock-port', action='store', type=int, default=10001, help='locking port')
    parser.add_argument('-s', '--sleep', action='store', type=int, default=0, help='random sleep on start')
    parser.add_argument('-V', '--version', action='store_true', help='output version info and exit')
    parser.add_argument('--opts', help='add extra options to skynet.bin')

    parser.add_argument(
        '-f', '--force', action='store_true',
        help='force installation even if actual version already installed'
    )
    parser.add_argument(
        '--force-update', action='store_true',
        help='force installation even if actual version already installed, but dont force services restart'
    )
    parser.add_argument(
        '-n', '--no-nice', action='store_true',
        help='avoid trying to be nice'
    )

    args = parser.parse_args()

    exc = None
    try:
        from gosky.__version__ import VERSION
        version_fail = False
    except Exception as ex:
        exc = ex
        VERSION = 'UNKNOWN'
        version_fail = True

    if args.version:
        try:
            sys.stdout.write('gosky v%s\n' % (VERSION, ))
            if version_fail:
                error(11, 'failed to get version: %s\n' % (exc, ))
        finally:
            raise SystemExit(0)

    info('Started gosky v%s', VERSION)
    uid = os.getuid()
    try:
        uname = pwd.getpwuid(uid).pw_name
    except:
        uname = str(uid)

    info('  User       : %s (%d)', uname, uid)
    info('  Lock port  : %s', args.lock_port)
    info('  Pid        : %d', os.getpid())
    info('  Platform   : %s', platform())

    if not args.no_nice:
        nice_change = check_nice_revertability()
        if nice_change != 0:
            os.nice(nice_change)

    deadline = time.time() + 300

    while True:
        try:
            hostname = get_hostname()
        except Exception as ex:
            timeleft = max(0, deadline - time.time())
            if timeleft > 0:
                info('  unable to get hostname (%s: %s)', type(ex).__name__, ex)
                info('  will try more for %d secs in 5s', timeleft)
                time.sleep(5)
            else:
                raise
        else:
            break

    ctx = {'args': args}
    ctx['stored'] = get_stored_context(hostname)
    ctx['stored']['platform'] = platform()
    ctx['stored']['hostname'] = hostname

    signal.alarm(7200)
    ctx['lock_sock_bind'], ctx['lock_sock'] = lock_on_port(args.lock_port, not args.sleep or args.force)

    info('  Locked on  : %s', ctx['lock_sock_bind'])

    random_sleep(args.sleep, ctx['lock_sock'])

    active_svn_url, active_svn_revision, active_version, active_state = get_active_info()

    info('Current:')
    info('  version    : %s [%s]', active_version or '(unknown)', active_svn_url)
    info('  state      : %s', active_state)
    info('  cfg_date   : %s', ctx['stored']['last_modified'] or '(none)')
    info('  config_hash: %s', ctx['stored']['config_hash'] or '(none)')
    info('  hostname   : %s', ctx['stored']['hostname'] or '(none)')

    if active_state not in ('ok', 'no_state'):
        info('Update will be forced: current skynet state is "%s"', active_state)
        args.force_update = True

    rinfo = get_freezed_release()
    if not rinfo:
        freezed = False
        rinfo = get_advertised_release(
            hostname,
            modified_since=ctx['stored']['last_modified'] if (not args.force and not args.force_update) else None,
            config_hash=ctx['stored']['config_hash'] if (not args.force and not args.force_update) else None
        )
        ctx['stored']['config_hash'] = rinfo['config_hash']
    else:
        # Reset stored last_update and config_hash if we have freezed release
        freezed = True
        ctx['stored']['last_update'] = None
        ctx['stored']['config_hash'] = None

    info('Advertized:')

    if 'last_modified' in rinfo:
        info('  cfg_date   : %s', rinfo['last_modified'])
        ctx['stored']['last_modified'] = rinfo['last_modified']
    else:
        ctx['stored']['last_modified'] = None

    if '@' in rinfo['svn_url'] and active_svn_revision:
        active_svn_url = '%s@%s' % (active_svn_url, active_svn_revision)

    info('  version    : %s [%s]', rinfo['version'] or '(unknown)', rinfo['svn_url'])
    info('  source     : %s' % (rinfo['advertizer'], ))

    if not freezed:
        info('  config_hash: %s' % (rinfo['config_hash'] or '(none)', ))

    if not args.force and not args.force_update and rinfo['svn_url'] == active_svn_url:
        save_stored_context(ctx['stored'])
        info('No need to update')
        raise SystemExit(0)

    info('')
    info('===' + ' PREPARE ' + '=' * (80 - len(' PREPARE ')))
    info('')

    skybin = download_skybin(rinfo['version'], rinfo['urls'], rinfo['md5'])
    if not skybin:
        error(11, 'Unable to download skynet binary.')

    info('')
    info('===' + ' INSTALL ' + '=' * (80 - len(' INSTALL ')))
    info('')

    code = run_skybin(ctx, skybin)
    if code in (0, 12):  # 12 -- installed, but not started cleanly
        save_stored_context(ctx['stored'])
        reshare_skybin(skybin)
    else:
        # On error -- do not save stored context, so full installation will be retried again
        pass

    if code > 0:
        error(code, 'Installation failed with exit code %d.', code)

    info('Installation completed successfully.')
