package main

import (
	"context"
	"errors"
	"flag"
	"fmt"
	"log"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/infra/goxcart/internal/app"
	"a.yandex-team.ru/infra/goxcart/internal/config"
)

func main() {
	os.Exit(run())
}

// Flags
const (
	usageTemplate = `Usage: %s [OPTIONS] CONFIG

Sidecar balancer configurator and supervisor.

CONFIG: path to YAML config file

OPTIONS:
`

	generateOnlyFlag    = "generate-only"
	generateOnlyDefault = false

	balancerConfigFlag    = "output"
	balancerConfigDefault = "config.lua"

	balancerBinFlag    = "balancer-bin"
	balancerBinDefault = "./balancer"

	balancerUserFlag    = "user"
	balancerUserDefault = "loadbase"

	balancerGroupFlag    = "group"
	balancerGroupDefault = "loadbase"
)

// Exit codes
const (
	_ = iota
	errorExitCode
	cliErrorExitCode
)

var logger = log.New(os.Stderr, "", log.Ldate|log.Ltime)

func cliError(fs *flag.FlagSet, err error) int {
	_, _ = fmt.Fprintf(os.Stderr, usageTemplate, os.Args[0])
	fs.PrintDefaults()
	_, _ = fmt.Fprintf(os.Stderr, "\nerror: %s\n", err)
	return cliErrorExitCode
}

// run executes application and returns exit code
func run() int {
	fs := flag.NewFlagSet(os.Args[0], flag.ContinueOnError)

	generateOnly := fs.Bool(generateOnlyFlag, generateOnlyDefault, "only generate balancer config")
	balancerConfig := fs.String(balancerConfigFlag, balancerConfigDefault,
		"output path for generated balancer lua config")
	balancerBin := fs.String(balancerBinFlag, balancerBinDefault, "path to balancer executable")

	balancerUser := fs.String(balancerUserFlag, balancerUserDefault, "set user to run balancer")
	balancerGroup := fs.String(balancerGroupFlag, balancerGroupDefault, "set group to run balancer")
	if (*balancerUser != balancerUserDefault) && (*balancerGroup == balancerGroupDefault) {
		balancerGroup = balancerUser
	}

	switch err := fs.Parse(os.Args[1:]); err {
	case nil:
	case flag.ErrHelp:
		return 0
	default:
		return cliError(fs, err)
	}

	if n := fs.NArg(); n != 1 {
		return cliError(fs, fmt.Errorf("expected 1 positional argument, got %d", n))
	}

	configPath := fs.Arg(0)
	if configPath == "" {
		return cliError(fs, errors.New("emtpy CONFIG"))
	}

	cfg, err := config.ReadYAML(configPath)
	if err != nil {
		log.Printf("config: %s", err)
		return errorExitCode
	}

	sigCh := make(chan os.Signal, 1)
	signal.Notify(sigCh, syscall.SIGINT, syscall.SIGTERM) // TODO: do not notify on SIGINT

	appl := &app.App{
		GenerateOnly:       *generateOnly,
		Config:             cfg,
		BalancerConfigPath: *balancerConfig,
		BalancerBinPath:    *balancerBin,
		Logger:             logger,
		User:               *balancerUser,
		Group:              *balancerGroup,
	}

	ctx, cancelCtx := context.WithCancel(context.Background())

	go func() {
		gotSig := <-sigCh
		logger.Printf("Got signal %q, gracefully stopping...", gotSig)
		cancelCtx()
	}()
	if err := appl.Start(ctx); err != nil {
		logger.Printf("error: %s", err)
		if ec := appl.BalancerExitCode(); ec != -1 {
			return ec
		}
		return errorExitCode
	}
	return 0
}
