package app

import (
	"context"
	"errors"
	"fmt"
	"log"
	"net"
	"os"
	"os/exec"
	"syscall"
	"time"

	"a.yandex-team.ru/infra/goxcart/internal/balancer"
	"a.yandex-team.ru/infra/goxcart/internal/config"
	"a.yandex-team.ru/infra/goxcart/internal/env"
	"a.yandex-team.ru/infra/goxcart/internal/unistats"
	"a.yandex-team.ru/infra/goxcart/pkg/watch"
	"golang.org/x/sync/errgroup"
	"golang.org/x/sys/unix"
)

// App implements the main logic of program.
type App struct {
	GenerateOnly       bool
	Config             *config.Config
	BalancerConfigPath string
	BalancerBinPath    string
	Logger             *log.Logger
	User               string
	Group              string

	balancer *balancer.Balancer
	unistats *unistats.Server
}

// Start starts an application and returns an error which can be caused by:
//   * error from reading/generating config files
//   * context cancellation
//   * BalancerExitedWithCode which means that the reason to application shutdown
//     was caused by termination of balancer process
//   * UnistatsServerError which means that an error occurred in unistats server
func (a *App) Start(ctx context.Context) error {
	if a.GenerateOnly {
		return a.Config.GenLuaConfig(a.BalancerConfigPath)
	}

	if !env.IsRoot() {
		return errors.New("goxcart must have root permissions. Please, run with sudo")
	}

	balancerUID, balancerGID, err := getBalancerCredentials(a.User, a.Group)
	if err != nil {
		return fmt.Errorf("unable to get balancer credentials: %w", err)
	}

	if err := a.Config.GenLuaConfig(a.BalancerConfigPath); err != nil {
		return fmt.Errorf("unable to generate balancer config: %w", err)
	}

	if err := a.createEnv(); err != nil {
		return fmt.Errorf("unable to create environment: %w", err)
	}
	defer a.cleanupEnv()

	ctx, cancelCtx := context.WithCancel(ctx)
	g, gCtx := errgroup.WithContext(ctx)
	defer g.Wait()
	defer cancelCtx()

	a.unistats = unistats.NewServer(&a.Config.UniStat.Addr, a.initStatSources(gCtx, g), a.Logger)
	g.Go(func() error {
		return a.unistats.Start(gCtx)
	})

	a.balancer = balancer.New(a.BalancerBinPath, a.BalancerConfigPath)
	a.balancer.Stdout = os.Stdout
	a.balancer.Stderr = os.Stderr
	a.balancer.SysProcAttr = &syscall.SysProcAttr{
		Pdeathsig: 15,
		Credential: &syscall.Credential{
			Uid: balancerUID,
			Gid: balancerGID,
		},
		AmbientCaps: []uintptr{unix.CAP_NET_BIND_SERVICE},
	}
	if err := a.balancer.Start(gCtx); err != nil {
		return err
	}
	g.Go(func() error {
		err := a.balancer.Wait()
		if err != nil {
			return err
		}
		// nil means command finished successfully (with exit code 0)
		// but errgroup.Group needs not nil error to indicate that
		// something went wrong, so we explicitly make it here
		return &exec.ExitError{ProcessState: a.balancer.Cmd.ProcessState}
	})

	return g.Wait()
}

func (a *App) BalancerPID() int {
	if a.balancer == nil || a.balancer.Process == nil {
		return -1
	}
	return a.balancer.Process.Pid
}

// BalancerExitCode returns exit code of Balancer
// or -1 if balancer has not been started or if was terminated by signal
func (a *App) BalancerExitCode() int {
	if a.balancer == nil {
		return -1
	}
	return a.balancer.ProcessState.ExitCode()
}

// initStatSources returns a list of Sources for unistats server
func (a *App) initStatSources(ctx context.Context, g *errgroup.Group) []unistats.Source {
	var cfgAddrs []net.TCPAddr // TODO: filter unique
	for _, b := range a.Config.Balancers {
		cfgAddrs = append(cfgAddrs, b.LocalEndpoint.Addr)
	}
	balancerUnistats := &watch.Unistat{URL: "http://" + net.JoinHostPort(
		config.BalancerUnistatIP, config.BalancerUnistatPort,
	) + "/unistat"}
	configAddrsCounter := &unistats.ConfigAddrsCounter{Addrs: cfgAddrs}
	aliveAddrsCounter := &unistats.AliveAddrsCounter{Addrs: cfgAddrs}
	dummyAddrsCounter := &unistats.IfaceAddrsCounter{Iface: &Iface}

	g.Go(func() error {
		return watch.UpdateEvery(ctx, 5*time.Second, []watch.Updater{
			aliveAddrsCounter,
			dummyAddrsCounter,
			balancerUnistats,
		}, a.Logger)
	})

	return []unistats.Source{
		configAddrsCounter,
		aliveAddrsCounter,
		dummyAddrsCounter,
		balancerUnistats,
	}
}
