package app

import (
	"fmt"
	"net"
	"os/user"
	"strconv"

	"a.yandex-team.ru/infra/goxcart/internal/env"
	"github.com/vishvananda/netlink"
)

// Iface is a dummy interface for adding addrs to.
var Iface = netlink.Dummy{
	LinkAttrs: netlink.LinkAttrs{
		TxQLen: -1,
		Name:   "dummy55",
	},
}

// IPv6Mask128 is used to add only one IPv6 address to interface.
// Since the length of an IPv6 addr is 128.
var IPv6Mask128 = net.IPMask{
	0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff,
}

// createEnv creates dummy interface and adds all addrs from config to it.
func (a *App) createEnv() error {
	if err := env.CreateInterface(&Iface); err != nil {
		return fmt.Errorf("unable to create interface: %w", err)
	}
	a.Logger.Printf("Interface %q was created", Iface.Name)

	addrs := make(map[string]struct{}, len(a.Config.Balancers))
	for _, balancer := range a.Config.Balancers {
		// do not add same addrs if ports are different
		addrStr := balancer.LocalEndpoint.Addr.String()
		if _, found := addrs[addrStr]; found {
			continue
		}
		addrs[addrStr] = struct{}{}

		if err := netlink.AddrAdd(&Iface, &netlink.Addr{
			IPNet: &net.IPNet{
				IP:   balancer.LocalEndpoint.Addr.IP,
				Mask: IPv6Mask128,
			},
		}); err != nil {
			return fmt.Errorf("unable to add addr %q to interface %q: %w",
				balancer.LocalEndpoint.Addr.IP,
				Iface.Name,
				err)
		}
		a.Logger.Printf("Address %q was added to interface %q", balancer.LocalEndpoint.Addr.IP, Iface.Name)
	}
	return nil
}

// cleanupEnv deletes the interface.
// It leads to requests are going to be routed through network which is better than
// being unable to send any requests to addrs on interface in case of existence
// an L3 balancer serving the same addresses.
func (a *App) cleanupEnv() error {
	if err := netlink.LinkDel(&Iface); err != nil {
		return fmt.Errorf("unable to delete interface %q: %w", Iface.Name, err)
	}
	a.Logger.Printf("Interface %q was deleted", Iface.Name)
	return nil
}

func getBalancerCredentials(usr, grp string) (UID, GID uint32, err error) {
	u, err := user.Lookup(usr)
	if err != nil {
		return 0, 0, fmt.Errorf("unable to find user %q: %w", usr, err)
	}
	g, err := user.LookupGroup(grp)
	if err != nil {
		return 0, 0, fmt.Errorf("unable to find group %q: %w", grp, err)
	}

	uid, err := strconv.Atoi(u.Uid)
	if err != nil {
		return 0, 0, fmt.Errorf("unable to parse UID %q: %w", u.Uid, err)
	}
	gid, err := strconv.Atoi(g.Gid)
	if err != nil {
		return 0, 0, fmt.Errorf("unable to parse GID %q: %w", g.Gid, err)
	}

	return uint32(uid), uint32(gid), nil
}
