package config

import (
	"net"
	"testing"

	"github.com/stretchr/testify/require"
	"gopkg.in/yaml.v2"
)

func Test_rawAddr_UnmarshalYAML(t *testing.T) {
	tests := []struct {
		name     string
		defaults net.TCPAddr
		data     string
		wantRes  net.TCPAddr
		wantErr  bool
	}{
		{
			name:    "empty without defaults",
			data:    "",
			wantErr: false,
		},
		{
			name: "empty with defaults",
			defaults: net.TCPAddr{
				IP:   net.ParseIP("fc00::1"),
				Port: 8080,
			},
			data: "",
			wantRes: net.TCPAddr{
				IP:   net.ParseIP("fc00::1"),
				Port: 8080,
			},
		},
		{
			name:    "invalid ip",
			data:    `ip: invalid`,
			wantErr: true,
		},
		{
			name:    "invalid port",
			data:    `port: -1`,
			wantErr: true,
		},
		{
			name: "override default ip",
			defaults: net.TCPAddr{
				IP:   net.ParseIP("fc00::1"),
				Port: 8080,
			},
			data: `ip: fc00::2`,
			wantRes: net.TCPAddr{
				IP:   net.ParseIP("fc00::2"),
				Port: 8080,
			},
		},
		{
			name: "override default port",
			defaults: net.TCPAddr{
				IP:   net.ParseIP("fc00::1"),
				Port: 8080,
			},
			data: `port: 8081`,
			wantRes: net.TCPAddr{
				IP:   net.ParseIP("fc00::1"),
				Port: 8081,
			},
		},
		{
			name: "override all defaults",
			defaults: net.TCPAddr{
				IP:   net.ParseIP("fc00::1"),
				Port: 8080,
			},
			data: `ip: fc00::2
port: 8081`,
			wantRes: net.TCPAddr{
				IP:   net.ParseIP("fc00::2"),
				Port: 8081,
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			ra := rawAddr(tt.defaults)

			if err := ra.UnmarshalYAML(func(i interface{}) error {
				return yaml.UnmarshalStrict([]byte(tt.data), i)
			}); tt.wantErr {
				require.Error(t, err)
				return
			} else {
				require.NoError(t, err)
			}
			require.EqualValues(t, tt.wantRes, ra)
		})
	}
}
