package config

import (
	"net"
	"testing"

	"github.com/stretchr/testify/require"
	"gopkg.in/yaml.v2"
)

func TestBalancers_UnmarshalYAML(t *testing.T) {
	tests := []struct {
		name    string
		data    string
		wantRes Balancers
		wantErr bool
	}{
		{
			name:    "empty",
			data:    "",
			wantRes: Balancers{},
		},
		{
			name: "one balancer",
			data: `b1:`,
			wantRes: Balancers{"b1": {
				ID:   "b1",
				Spec: DefaultSpec,
			}},
		},
		{
			name: "repeated ids",
			data: `b1:
  local_endpoint:
    ip: fc00::1
    port: 8080
b1:
  local_endpoint:
    ip: fc00::2
    port: 8081`,
			wantErr: true,
		},
		{
			name: "multiple",
			data: `b1:
  local_endpoint:
    ip: fc00::1
    port: 8080
b2:
  local_endpoint:
    ip: fc00::2
    port: 8081`,
			wantRes: Balancers{
				"b1": &Balancer{
					ID: "b1",
					LocalEndpoint: LocalEndpoint{
						Addr: net.TCPAddr{
							IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
							Port: 8080,
						},
					},
					Spec: DefaultSpec,
				},
				"b2": &Balancer{
					ID: "b2",
					LocalEndpoint: LocalEndpoint{
						Addr: net.TCPAddr{
							IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2},
							Port: 8081,
						},
					},
					Spec: DefaultSpec,
				},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			b := make(Balancers)
			if err := b.UnmarshalYAML(func(i interface{}) error {
				return yaml.UnmarshalStrict([]byte(tt.data), i)
			}); tt.wantErr {
				require.Error(t, err)
				return
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tt.wantRes, b)
		})
	}
}
