package config

import (
	"fmt"
	"os"
	"strings"

	"a.yandex-team.ru/library/go/valid"
	"gopkg.in/yaml.v2"
)

// ReadYAML reads Config from YAML file with path provided and
// validates it.
func ReadYAML(path string) (*Config, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, fmt.Errorf("unable to open file: %w", err)
	}
	defer f.Close()

	// init in case of empty config
	cfg := DefaultConfig
	dec := yaml.NewDecoder(f)
	dec.SetStrict(true)
	if err := dec.Decode(&cfg); err != nil {
		return nil, fmt.Errorf("unable to parse config: %w", err)
	}

	if err := valid.Struct(vCtx, &cfg); err != nil {
		if errs, ok := err.(valid.Errors); ok {
			var b strings.Builder
			// TODO: maybe do this somewhere upper?
			for _, err := range errs {
				_, _ = b.WriteString(fmt.Sprintf("%+v; ", err))
			}
			return &cfg, fmt.Errorf("validation failed: %s", &b)
		}
		return &cfg, fmt.Errorf("validation failed: %w", err)
	}

	return &cfg, nil
}

func ReadAndGen(in, out string) (*Config, error) {
	cfg, err := ReadYAML(in)
	if err != nil {
		return nil, err
	}
	return cfg, cfg.GenLuaConfig(out)
}

var DefaultConfig = Config{
	StateDirectory: "/dev/shm/balancer-state",
	UniStat:        DefaultUniStat,
}

const DefaultStateDirectory = "/dev/shm/balancer-state"

// Config is a root structure for goxcart config
type Config struct {
	StateDirectory   string           `yaml:"state_directory"`
	UniStat          UniStat          `yaml:"unistat"`
	ServiceDiscovery ServiceDiscovery `yaml:"service_discovery"`
	Balancers        Balancers        `yaml:"balancers"`
}

func (c *Config) UnmarshalYAML(unmarshal func(interface{}) error) error {
	type rawConfig Config
	tmp := rawConfig(Config{
		StateDirectory:   DefaultStateDirectory,
		UniStat:          DefaultUniStat,
		ServiceDiscovery: DefaultServiceDiscovery,
	})
	if err := unmarshal(&tmp); err != nil {
		return err
	}
	c.StateDirectory = tmp.StateDirectory
	c.UniStat = tmp.UniStat
	c.ServiceDiscovery = tmp.ServiceDiscovery
	c.Balancers = tmp.Balancers
	return nil
}

// Validate checks for required fields and sets defaults
func (c *Config) Validate(_ *valid.ValidationCtx) (bool, error) {
	// unique pairs <host, port>
	hostPorts := map[string]struct{}{
		c.UniStat.Addr.String(): {}, // add unistat
	}
	// add validator
	balancerValidators := []func(*Balancer) error{func(b *Balancer) error {
		hostPort := b.LocalEndpoint.Addr.String()
		if _, found := hostPorts[hostPort]; found {
			return &DuplicateFound{
				where: "HOST:PORT",
				what:  hostPort,
			}
		}
		hostPorts[hostPort] = struct{}{}
		return nil
	}}

	if c.UniStat.Addr.IP.IsUnspecified() {
		// In case of unistat addr is [::]:UNISTAT_PORT, then if one of balancers wants to serve
		// <dummy addr>:UNISTAT_PORT, balancer will not be able to bind to this port
		// since this port is already in use on all addresses by unistat server.
		// However, this is not fatal error for balancer and it will continue its execution and log "unable to bind
		// error. We would not be able to detect such behaviour.
		// So we must explicitly forbid anyone to use this port.
		balancerValidators = append(balancerValidators, func(b *Balancer) error {
			if b.LocalEndpoint.Addr.Port == c.UniStat.Addr.Port {
				return fmt.Errorf("port %d is already in use by \"unistat\" on \"::\"", c.UniStat.Addr.Port)
			}
			return nil
		})
	}

	return true, c.validateBalancers(balancerValidators...)
}

func (c *Config) validateBalancers(validators ...func(balancers *Balancer) error) error {
	for id, b := range c.Balancers {
		for _, v := range validators {
			if err := v(b); err != nil {
				return fmt.Errorf("balancer %q: %w", id, err)
			}
		}
	}
	return nil
}
