package config

import (
	"io/ioutil"
	"net"
	"path"
	"path/filepath"
	"testing"

	"a.yandex-team.ru/library/go/valid"
	"github.com/stretchr/testify/require"
)

const examplesDir = "testdata"

func TestExamples(t *testing.T) {
	matches, err := filepath.Glob(path.Join(examplesDir, "*.yaml"))
	require.NoError(t, err)
	for _, m := range matches {
		t.Run(path.Base(m), func(t *testing.T) {
			cfg, err := ReadYAML(m)
			require.NoError(t, err)
			require.NoError(t, cfg.genLua(ioutil.Discard))
		})
	}
}

func TestConfig_Validate(t *testing.T) {
	tests := []struct {
		name         string
		cfg          Config
		wantFailFast bool
		wantErr      bool
	}{
		{
			name: "unistat ip is unspecified and port is reused",
			cfg: Config{
				UniStat: UniStat{
					Addr: net.TCPAddr{
						IP:   net.IPv6unspecified,
						Port: 8765,
					},
				},
				Balancers: Balancers{
					"id1": &Balancer{
						LocalEndpoint: LocalEndpoint{
							Addr: net.TCPAddr{
								IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
								Port: 8765,
							},
						},
					},
				},
			},
			wantErr: true,
		},
		{
			name: "unistat ip is unspecified and port is not reused",
			cfg: Config{
				UniStat: UniStat{
					Addr: net.TCPAddr{
						IP:   net.IPv6unspecified,
						Port: 8765,
					},
				},
				Balancers: Balancers{
					"id1": &Balancer{
						LocalEndpoint: LocalEndpoint{
							Addr: net.TCPAddr{
								IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
								Port: 5678,
							},
						},
					},
				},
			},
			wantErr: false,
		},
		{
			name: "<ip, port> pair is reused",
			cfg: Config{
				Balancers: Balancers{
					"id1": &Balancer{
						LocalEndpoint: LocalEndpoint{
							Addr: net.TCPAddr{
								IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
								Port: 8765,
							},
						},
					},
					"id2": &Balancer{
						LocalEndpoint: LocalEndpoint{
							Addr: net.TCPAddr{
								IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
								Port: 8765,
							},
						},
					},
				},
			},
			wantErr: true,
		},
		{
			name: "all <ip, port> pairs are unique",
			cfg: Config{
				Balancers: Balancers{
					"id1": &Balancer{
						LocalEndpoint: LocalEndpoint{
							Addr: net.TCPAddr{
								IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
								Port: 8765,
							},
						},
					},
					"id2": &Balancer{
						LocalEndpoint: LocalEndpoint{
							Addr: net.TCPAddr{
								IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
								Port: 5678,
							},
						},
					},
					"id3": &Balancer{
						LocalEndpoint: LocalEndpoint{
							Addr: net.TCPAddr{
								IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2},
								Port: 8765,
							},
						},
					},
					"id4": &Balancer{
						LocalEndpoint: LocalEndpoint{
							Addr: net.TCPAddr{
								IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2},
								Port: 5678,
							},
						},
					},
				},
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			proceed, err := tt.cfg.Validate(valid.NewValidationCtx())
			if tt.wantErr {
				require.Error(t, err)
				return
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, !tt.wantFailFast, proceed)
		})
	}
}
