package config

import (
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"gopkg.in/yaml.v2"
)

func TestDuration_UnmarshalYAML(t *testing.T) {
	tests := []struct {
		name    string
		data    string
		wantRes time.Duration
		wantErr bool
	}{
		{
			name:    "empty",
			data:    "",
			wantErr: true,
		},
		{
			name:    "not duration",
			data:    "abc",
			wantErr: true,
		},
		{
			name:    "zero without unit",
			data:    "0",
			wantRes: 0,
		},
		{
			name:    "missing unit",
			data:    "10",
			wantErr: true,
		},
		{
			name:    "seconds only",
			data:    "100s",
			wantRes: 100 * time.Second,
		},
		{
			name:    "repeated units",
			data:    "33ms999ms",
			wantRes: 33*time.Millisecond + 999*time.Millisecond,
		},
		{
			name:    "mixed",
			data:    "1h20m33s999ms",
			wantRes: time.Hour + 20*time.Minute + 33*time.Second + 999*time.Millisecond,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			var d Duration
			if err := d.UnmarshalYAML(func(i interface{}) error {
				return yaml.UnmarshalStrict([]byte(tt.data), i)
			}); tt.wantErr {
				require.Error(t, err)
				return
			} else {
				require.NoError(t, err)
			}
			require.EqualValues(t, tt.wantRes, d)
		})
	}
}

func durationPtr(d time.Duration) *Duration {
	dd := Duration(d)
	return &dd
}
