package config

import (
	"testing"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/library/go/valid"
	"github.com/stretchr/testify/require"
)

func TestDynamic_Validate(t *testing.T) {
	tests := []struct {
		name    string
		dynamic *Dynamic
		wantErr bool
	}{
		{
			name:    "requires itself",
			dynamic: nil,
			wantErr: true,
		},
		{
			name: "requires MaxPessimizedShare",
			dynamic: &Dynamic{
				MaxPessimizedShare:    nil,
				MinPessimizationCoeff: ptr.Float32(0),
				WeightIncreaseStep:    ptr.Float32(0),
				HistoryInterval:       durationPtr(0),
				BackendsName:          "test",
			},
			wantErr: true,
		},
		{
			name: "requires MinPessimizationCoeff",
			dynamic: &Dynamic{
				MaxPessimizedShare:    ptr.Float32(0),
				MinPessimizationCoeff: nil,
				WeightIncreaseStep:    ptr.Float32(0),
				HistoryInterval:       durationPtr(0),
				BackendsName:          "test",
			},
			wantErr: true,
		},
		{
			name: "requires WeightIncreaseStep",
			dynamic: &Dynamic{
				MaxPessimizedShare:    ptr.Float32(0),
				MinPessimizationCoeff: ptr.Float32(0),
				WeightIncreaseStep:    nil,
				HistoryInterval:       durationPtr(0),
				BackendsName:          "test",
			},
			wantErr: true,
		},
		{
			name: "requires HistoryInterval",
			dynamic: &Dynamic{
				MaxPessimizedShare:    ptr.Float32(0),
				MinPessimizationCoeff: ptr.Float32(0),
				WeightIncreaseStep:    ptr.Float32(0),
				HistoryInterval:       nil,
				BackendsName:          "test",
			},
			wantErr: true,
		},
		{
			name: "requires BackendsName",
			dynamic: &Dynamic{
				MaxPessimizedShare:    ptr.Float32(0),
				MinPessimizationCoeff: ptr.Float32(0),
				WeightIncreaseStep:    ptr.Float32(0),
				HistoryInterval:       durationPtr(0),
				BackendsName:          "",
			},
			wantErr: true,
		},
		{
			name: "full is ok",
			dynamic: &Dynamic{
				MaxPessimizedShare:    ptr.Float32(0),
				MinPessimizationCoeff: ptr.Float32(0),
				WeightIncreaseStep:    ptr.Float32(0),
				HistoryInterval:       durationPtr(0),
				BackendsName:          "test",
			},
			wantErr: false,
		},
		{
			name: "active is required unless it is empty",
			dynamic: &Dynamic{
				MaxPessimizedShare:    ptr.Float32(0),
				MinPessimizationCoeff: ptr.Float32(0),
				WeightIncreaseStep:    ptr.Float32(0),
				HistoryInterval:       durationPtr(0),
				BackendsName:          "test",
				Active:                &Active{},
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if err := valid.Struct(vCtx, tt.dynamic); tt.wantErr {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
		})
	}
}

func TestActive_Validate(t *testing.T) {
	tests := []struct {
		name    string
		active  *Active
		wantErr bool
	}{
		{
			name: "requires UseBackendWeight",
			active: &Active{
				UseBackendWeight:         nil,
				WeightNormalizationCoeff: ptr.Float32(0),
				Request:                  "test",
				Delay:                    durationPtr(0),
			},
			wantErr: true,
		},
		{
			name: "requires WeightNormalizationCoeff",
			active: &Active{
				UseBackendWeight:         ptr.Bool(false),
				WeightNormalizationCoeff: nil,
				Request:                  "test",
				Delay:                    durationPtr(0),
			},
			wantErr: true,
		},
		{
			name: "requires Request",
			active: &Active{
				UseBackendWeight:         ptr.Bool(false),
				WeightNormalizationCoeff: ptr.Float32(0),
				Request:                  "",
				Delay:                    durationPtr(0),
			},
			wantErr: true,
		},
		{
			name: "requires Delay",
			active: &Active{
				UseBackendWeight:         ptr.Bool(false),
				WeightNormalizationCoeff: ptr.Float32(0),
				Request:                  "test",
				Delay:                    nil,
			},
			wantErr: true,
		},
		{
			name: "full is ok",
			active: &Active{
				UseBackendWeight:         ptr.Bool(false),
				WeightNormalizationCoeff: ptr.Float32(0),
				Request:                  "test",
				Delay:                    durationPtr(0),
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if err := valid.Struct(vCtx, tt.active); tt.wantErr {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
		})
	}
}
