package config

import (
	"net"
	"testing"

	"github.com/stretchr/testify/require"
	"gopkg.in/yaml.v2"
)

func TestLocalEndpoint_UnmarshalYAML(t *testing.T) {
	tests := []struct {
		name    string
		data    string
		wantRes LocalEndpoint
		wantErr bool
	}{
		{
			name:    "empty",
			data:    ``,
			wantErr: false,
		},
		{
			name:    "invalid ip",
			data:    "ip: abc",
			wantErr: true,
		},
		{
			name: "ip only",
			data: "ip: fc00::1",
			wantRes: LocalEndpoint{
				Addr: net.TCPAddr{
					IP: net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
				},
			},
		},
		{
			name: "port only",
			data: "port: 8181",
			wantRes: LocalEndpoint{
				Addr: net.TCPAddr{
					Port: 8181,
				},
			},
		},
		{
			name: "everything",
			data: `ip: fc00::1
port: 8181
fqdn: some.host.net`,
			wantRes: LocalEndpoint{
				Addr: net.TCPAddr{
					IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
					Port: 8181,
				},
				FQDN: "some.host.net",
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			var le LocalEndpoint
			if err := yaml.UnmarshalStrict([]byte(tt.data), &le); tt.wantErr {
				require.Error(t, err)
				return
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tt.wantRes, le)
		})
	}
}
