package config

import (
	"fmt"
	"io"
	"os"
	"text/template"
)

const (
	BalancerAdminIP     = "localhost"
	BalancerAdminPort   = "8081"
	BalancerUnistatIP   = "localhost"
	BalancerUnistatPort = "8082"
	luaConfigTmpl       = `instance = {
  state_directory = {{printf "%q" .StateDirectory}};

  addrs = { {{- range .Balancers}}
    {{printf "{ ip = %q; port = %d; };" .LocalEndpoint.Addr.IP .LocalEndpoint.Addr.Port}}{{end}}
  }; -- addrs

  admin_addrs = {
    { ip = "` + BalancerAdminIP + `"; port = ` + BalancerAdminPort + `; };
  }; -- admin_addrs

  unistat = {
    addrs = {
      { ip = "` + BalancerUnistatIP + `"; port = ` + BalancerUnistatPort + `; };
    }; -- addrs
  }; -- unistat

  {{with $sd := .ServiceDiscovery}}sd = {
    host             = {{printf "%q" $sd.Host}};
    port             = {{printf "%d" $sd.Port}};
    connect_timeout  = {{printf "%q" $sd.ConnectTimeout.MarshalLua}};
    request_timeout  = {{printf "%q" $sd.RequestTimeout.MarshalLua}};
    cache_dir        = {{printf "%q" $sd.CacheDir}};
    log              = {{printf "%q" $sd.Log}};
    client_name      = {{printf "%q" $sd.ClientName}};
    update_frequency = {{printf "%q" $sd.UpdateFrequency.MarshalLua}};
  }; -- sd{{end}}

  ipdispatch = {
    admin = {
      ip   = "localhost";
      port = 8081;

      http = {
        maxlen = 65536;
        maxreq = 65536;
        admin  = {};
      }; -- http
    }; -- admin{{range $bID, $balancer := .Balancers}}
    {{with $name := $bID}}{{printf "[%q]" $name}} = {
      ip       = {{printf "%q" $balancer.LocalEndpoint.Addr.IP}};
      port     = {{printf "%d" $balancer.LocalEndpoint.Addr.Port}};

      errorlog = {
        log = "errorlog.log";
        accesslog = {
          report = {
            uuid   = {{printf "%q" $bID}};
            ranges = "7ms,11ms,17ms,26ms,39ms,58ms,87ms,131ms,197ms,296ms,444ms,666ms,1000ms,1500ms,2250ms,3375ms,5062ms,7593ms,11390ms,17085ms";
            events = {
              stats = "report";
            }; -- events
            balancer2 = {
			  attempts = {{printf "%d" $balancer.Spec.Attempts}};
              sd = {
                endpoint_sets = { {{range $YPID, $YPEndpointSet := $balancer.Spec.YPEndpointSets}}
                  {{printf "{ cluster_name = %q; endpoint_set_id = %q; };" $YPEndpointSet.Cluster $YPEndpointSet.ID}}
                {{end}} }; -- endpoint_sets

                proxy_options = {
                  connect_timeout = {{printf "%q" $balancer.Spec.ConnectTimeout.MarshalLua}};
                  backend_timeout = {{printf "%q" $balancer.Spec.BackendTimeout.MarshalLua}};
                }; -- proxy_options

                {{with $balancer.Spec.Dynamic}}dynamic = {
                  max_pessimized_share    = {{.MaxPessimizedShare}};
                  min_pessimization_coeff = {{.MinPessimizationCoeff}};
                  weight_increase_step    = {{.WeightIncreaseStep}};
                  history_interval        = {{printf "%q" .HistoryInterval.MarshalLua}};
                  backends_name           = {{printf "%q" .BackendsName}};{{with .TCPCheck}}
				  tcp_check               = {{.}};{{end}}{{with .Active}}
                  active = {
                    use_backend_weight         = {{.UseBackendWeight}};
                    weight_normalization_coeff = {{.WeightNormalizationCoeff}};
                    request                    = {{printf "%q" .Request}};
                    delay                      = {{printf "%q" .Delay.MarshalLua}};
                  }; -- active{{end}}
                }; -- dynamic{{end}}
              }; -- sd
            }; -- balancer2
          }; -- report
        }; -- accesslog
      }; -- errorlog
    }; -- {{$name}}{{end}}{{end}}
  }; -- ipdispatch
}; -- instance
`
)

var tmpl, _ = template.New("config").Parse(luaConfigTmpl)

// genLua generates balancer Lua config from template and writes it to the  writer.
func (c *Config) genLua(w io.Writer) error {
	if err := tmpl.Execute(w, c); err != nil {
		return fmt.Errorf("unable to execute template: %w", err)
	}
	return nil
}

// GenLuaConfig creates balancer Lua config in given path.
func (c *Config) GenLuaConfig(outputPath string) error {
	f, err := os.Create(outputPath)
	if err != nil {
		return fmt.Errorf("unable to create file: %w", err)
	}
	defer f.Close()

	return c.genLua(f)
}
