package config

import (
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"gopkg.in/yaml.v2"
)

func TestServiceDiscovery_UnmarshalYAML(t *testing.T) {
	tests := []struct {
		name    string
		data    string
		wantRes ServiceDiscovery
		wantErr bool
	}{
		{
			name:    "empty",
			data:    ``,
			wantRes: DefaultServiceDiscovery,
		},
		{
			name: "required only",
			data: `client_name: test`,
			wantRes: ServiceDiscovery{
				Host:            DefaultServiceDiscovery.Host,
				Port:            DefaultServiceDiscovery.Port,
				UpdateFrequency: DefaultServiceDiscovery.UpdateFrequency,
				ConnectTimeout:  DefaultServiceDiscovery.ConnectTimeout,
				RequestTimeout:  DefaultServiceDiscovery.RequestTimeout,
				ClientName:      "test",
				CacheDir:        DefaultServiceDiscovery.CacheDir,
				Log:             DefaultServiceDiscovery.Log,
			},
		},
		{
			name: "override all defaults",
			data: `
host: test.yandex.net
port: 8181
update_frequency: 2m
connect_timeout: 200ms
request_timeout: 5s
client_name: test
cache_dir: "./test_dir"
log: "/test/sd_log"`,
			wantRes: ServiceDiscovery{
				Host:            "test.yandex.net",
				Port:            uint16(8181),
				UpdateFrequency: Duration(2 * time.Minute),
				ConnectTimeout:  Duration(200 * time.Millisecond),
				RequestTimeout:  Duration(5 * time.Second),
				ClientName:      "test",
				CacheDir:        "./test_dir",
				Log:             "/test/sd_log",
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			sd := DefaultServiceDiscovery
			if err := sd.UnmarshalYAML(func(i interface{}) error {
				return yaml.UnmarshalStrict([]byte(tt.data), i)
			}); tt.wantErr {
				require.Error(t, err)
				return
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tt.wantRes, sd)
		})
	}
}
