package config

import (
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"gopkg.in/yaml.v2"
)

func TestSpec_UnmarshalYAML(t *testing.T) {
	tests := []struct {
		name    string
		data    string
		wantRes Spec
		wantErr bool
	}{
		{
			name:    "empty",
			data:    ``,
			wantRes: DefaultSpec,
		},
		{
			name: "required only",
			data: `yp_endpointsets:
  - cluster: sas
    id: test
`,
			wantRes: Spec{
				YPEndpointSets: []YPEndpointSet{YPEndpointSet{Cluster: "sas", ID: "test"}},
				BackendTimeout: DefaultSpec.BackendTimeout,
				ConnectTimeout: DefaultSpec.ConnectTimeout,
				Attempts:       1,
			},
		},
		{
			name: "override defaults",
			data: `yp_endpointsets:
  - cluster: sas
    id: test
  - cluster: vla
    id: vla-test
backend_timeout: 10h
connect_timeout: 15s
attempts: 2`,
			wantRes: Spec{
				YPEndpointSets: []YPEndpointSet{YPEndpointSet{Cluster: "sas", ID: "test"}, YPEndpointSet{Cluster: "vla", ID: "vla-test"}},
				BackendTimeout: Duration(10 * time.Hour),
				ConnectTimeout: Duration(15 * time.Second),
				Attempts:       2,
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			var s Spec
			if err := s.UnmarshalYAML(func(i interface{}) error {
				return yaml.UnmarshalStrict([]byte(tt.data), i)
			}); tt.wantErr {
				require.Error(t, err)
				return
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tt.wantRes, s)
		})
	}
}
