package config

import (
	"net"
	"testing"

	"github.com/stretchr/testify/require"
	"gopkg.in/yaml.v2"
)

func TestUniStat_UnmarshalYAML(t *testing.T) {
	tests := []struct {
		name    string
		data    string
		wantRes UniStat
		wantErr bool
	}{
		{
			name:    "empty",
			data:    ``,
			wantRes: DefaultUniStat,
		},
		{
			name:    "invalid ip",
			data:    "ip: abc",
			wantErr: true,
		},
		{
			name: "ip only",
			data: "ip: fc00::1",
			wantRes: UniStat{
				Addr: net.TCPAddr{
					IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
					Port: DefaultUniStat.Addr.Port,
				},
			},
		},
		{
			name: "port only",
			data: "port: 8181",
			wantRes: UniStat{
				Addr: net.TCPAddr{
					IP:   DefaultUniStat.Addr.IP,
					Port: 8181,
				},
			},
		},
		{
			name: "ip and port",
			data: `ip: fc00::1
port: 8181`,
			wantRes: UniStat{
				Addr: net.TCPAddr{
					IP:   net.IP{0xfc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1},
					Port: 8181,
				},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			s := DefaultUniStat
			if err := s.UnmarshalYAML(func(i interface{}) error {
				return yaml.UnmarshalStrict([]byte(tt.data), i)
			}); tt.wantErr {
				require.Error(t, err)
				return
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tt.wantRes, s)
		})
	}
}
