package config

import (
	"fmt"
	"reflect"
	"strings"

	"a.yandex-team.ru/library/go/valid"
)

type Required string

func (e Required) Error() string {
	if e == "" {
		return "required"
	}
	return fmt.Sprintf("%q is required", string(e))
}

// required="" checks for value to be not zero for its type
// required="A.B.C+A.D" means that
// field C of field B of field A of current struct
// and field D of field A of current struct are required
func required(value reflect.Value, param string) error {
	if param == "" {
		if !value.IsValid() || value.IsZero() {
			return Required("")
		}
		return nil
	}
	if value.Kind() != reflect.Struct {
		return valid.ErrInvalidType
	}
	var errs valid.Errors
	for _, path := range strings.Split(param, "+") {
		var first, rest string
		switch bound := strings.IndexByte(path, '.'); bound {
		case -1: // first_level field
			first = path
		case 0: // begins with dot
			return valid.ErrBadParams
		default:
			first = path[:bound]
			rest = path[bound+1:]
		}
		switch err := required(value.FieldByName(first), rest); e := err.(type) {
		case nil:
		case Required:
			errs = append(errs, Required(first+string(e)))
		default:
			errs = append(errs, fmt.Errorf("%s: %w", first, err))
		}
	}
	switch len(errs) {
	case 0:
		return nil
	case 1:
		return errs[0]
	}
	return errs
}

var vCtx = valid.NewValidationCtx()

func init() {
	vCtx.Add("required", required)
}
