package config

import (
	"reflect"
	"testing"

	"github.com/stretchr/testify/require"
)

func Test_required(t *testing.T) {
	tests := []struct {
		name    string
		val     interface{}
		param   string
		wantErr bool
	}{
		{
			name:    "nil",
			val:     nil,
			wantErr: true,
		},
		{
			name:    "zero basic type",
			val:     0,
			wantErr: true,
		},
		{
			name:    "non-zero basic type",
			val:     1,
			wantErr: false,
		},
		{
			name:    "nil slice",
			val:     []int(nil),
			wantErr: true,
		},
		{
			name:    "nil map",
			val:     map[int]int(nil),
			wantErr: true,
		},
		{
			name:    "empty struct",
			val:     struct{}{},
			wantErr: true,
		},
		{
			name: "zero struct",
			val: struct {
				A int
			}{},
			wantErr: true,
		},
		{
			name:    "non-zero struct",
			val:     struct{ A int }{A: 1},
			wantErr: false,
		},
		{
			name: "struct with zero required first-level field",
			val: struct {
				A int
				B string
			}{
				B: "not zero",
			},
			param:   "A",
			wantErr: true,
		},
		{
			name: "struct with zero required deep-level field",
			val: struct {
				A struct {
					B int
					C string
				}
			}{
				A: struct {
					B int
					C string
				}{C: "not zero"},
			},
			param:   "A.B",
			wantErr: true,
		},
		{
			name: "struct with non-zero required deep-level field",
			val: struct {
				A struct {
					B int
					C string
				}
			}{
				A: struct {
					B int
					C string
				}{B: 1},
			},
			param:   "A.B",
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if err := required(reflect.ValueOf(tt.val), tt.param); tt.wantErr {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
		})
	}
}
