package env

import (
	"fmt"
	"syscall"

	"github.com/vishvananda/netlink"
)

// CreateInterface creates link iface.
// If error is nil, created interface will not have any addresses assigned.
// Existence of such an interface is not an error as it will be recreated.
func CreateInterface(iface netlink.Link) error {
	if err := netlink.LinkAdd(iface); err != nil {
		switch err {
		case syscall.EEXIST: // we need to recreate interface to not remove all addresses on it manually
			if err := netlink.LinkDel(iface); err != nil {
				return fmt.Errorf("unable to delete existing link: %w", err)
			}
			if err := netlink.LinkAdd(iface); err != nil {
				return fmt.Errorf("unable to add link after delete: %w", err)
			}
		default:
			return fmt.Errorf("unable to add link: %w", err)
		}
	}

	if err := netlink.LinkSetUp(iface); err != nil {
		return fmt.Errorf("unable to set link up: %w", err)
	}

	return nil
}
