package unistats

import (
	"context"
	"net"
	"sync"

	"a.yandex-team.ru/infra/goxcart/pkg/unistats"
	"github.com/vishvananda/netlink"
)

// Signal names
const (
	configAddrsSignalName = "goxcart-config-addrs_ammx"
	aliveAddrsSignalName  = "goxcart-alive-addrs_ammx"
	ifaceAddrsSignalName  = "goxcart-dummy-addrs_ammx"
)

// ConfigAddrsCounter is a Source which stores
// the count of different addresses in goxcart config.
type ConfigAddrsCounter struct {
	Addrs []net.TCPAddr
}

func (c *ConfigAddrsCounter) Unistats() unistats.UniStats {
	return unistats.UniStats{
		{configAddrsSignalName, len(c.Addrs)},
	}
}

// AliveAddrsCounter is a Source which watches after
// the count of alive addresses among specified in config.
type AliveAddrsCounter struct {
	Addrs []net.TCPAddr

	lastCount uint
	mu        sync.RWMutex
}

func (c *AliveAddrsCounter) String() string {
	return "alive addrs counter"
}

// Update recounts alive addresses
func (c *AliveAddrsCounter) Update(_ context.Context) error {
	var count uint
	for _, addr := range c.Addrs {
		if c.isAlive(&addr) {
			count++
		}
	}
	c.setCount(count)

	return nil
}

func (c *AliveAddrsCounter) Drop() {
	c.setCount(0)
}

// setCount sets current count in a thread-safe way
func (c *AliveAddrsCounter) setCount(count uint) {
	c.mu.RLock()
	defer c.mu.RUnlock()

	c.lastCount = count
}

// LastCount returns last stored count
func (c *AliveAddrsCounter) LastCount() uint {
	c.mu.Lock()
	defer c.mu.Unlock()

	return c.lastCount
}

func (c *AliveAddrsCounter) Unistats() unistats.UniStats {
	return unistats.UniStats{
		{aliveAddrsSignalName, c.LastCount()}, // TODO: check YASM spec if it is okay to return in such format
	}
}

// isAlive returns whether given address is alive or not via TCP handshake.
func (c *AliveAddrsCounter) isAlive(addr *net.TCPAddr) bool {
	conn, err := net.DialTCP(addr.Network(), nil, addr)
	if err != nil {
		return false
	}
	defer conn.Close()
	return true
}

// IfaceAddrsCounter is a Source which watches after
// the count of addresses assigned to an interface.
type IfaceAddrsCounter struct {
	Iface netlink.Link

	lastCount uint
	mu        sync.RWMutex
}

func (c *IfaceAddrsCounter) String() string {
	return "iface addrs counter"
}

// Updates updates the count of present addrs on interface.
// In case of error, count will be dropped to zero.
func (c *IfaceAddrsCounter) Update(_ context.Context) (err error) {
	addrs, err := netlink.AddrList(c.Iface, netlink.FAMILY_V6)
	if err != nil {
		return err
	}
	var notLinkLocalCount uint
	for _, a := range addrs {
		if a.IP.IsLinkLocalUnicast() {
			continue
		}
		notLinkLocalCount++
	}

	c.setCount(notLinkLocalCount)

	return nil
}

func (c *IfaceAddrsCounter) Drop() {
	c.setCount(0)
}

// setCount atomically sets the count
func (c *IfaceAddrsCounter) setCount(count uint) {
	c.mu.RLock()
	defer c.mu.RUnlock()

	c.lastCount = count
}

// LastCount return last updated count
func (c *IfaceAddrsCounter) LastCount() uint {
	c.mu.Lock()
	defer c.mu.Unlock()

	return c.lastCount
}

func (c *IfaceAddrsCounter) Unistats() unistats.UniStats {
	return unistats.UniStats{
		{ifaceAddrsSignalName, c.LastCount()},
	}
}
