package unistats

import (
	"context"
	"encoding/json"
	"log"
	"net"
	"net/http"

	"a.yandex-team.ru/infra/goxcart/pkg/unistats"
)

const (
	unistatEndpoint = "/unistat"
)

// Server is an HTTP server.
// It has the only valid endpoint (GET /unistats)
// which returns merged unistats from its sources
type Server struct {
	sources []Source
	logger  *log.Logger
	srv     *http.Server
}

type Source interface {
	Unistats() unistats.UniStats
}

// NewServer creates Server with handlers set.
// After Start() on returned server, it will listen on supplied TCPAddr
func NewServer(addr *net.TCPAddr, unistatSources []Source, logger *log.Logger) *Server {
	mux := http.NewServeMux()
	s := &Server{
		sources: unistatSources,
		logger:  logger,
		srv: &http.Server{
			Addr:    addr.String(),
			Handler: mux,
		},
	}
	mux.Handle(unistatEndpoint, s)
	return s
}

// Start starts server loop and returns an error in case:
//   * something went wrong inside HTTP server
//   * context was cancelled
func (s *Server) Start(ctx context.Context) error {
	go func() {
		<-ctx.Done()
		_ = s.srv.Shutdown(context.Background())
	}()
	if err := s.srv.ListenAndServe(); err != http.ErrServerClosed {
		return err
	}
	return nil
}

// ServeHTTP handles an incoming HTTP GET request and returns gathered unistats from all sources.
func (s *Server) ServeHTTP(rw http.ResponseWriter, req *http.Request) {
	if req.Method != http.MethodGet {
		http.NotFound(rw, req) // TODO: 405 method not allowed
		return
	}
	resp := make(unistats.UniStats, 0, len(s.sources))
	for _, src := range s.sources {
		resp = resp.Merge(src.Unistats())
	}

	if err := json.NewEncoder(rw).Encode(resp); err != nil {
		s.logger.Printf("unable to encode response: %s", err)
		return
	}
}
