package watch

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"sync"

	"a.yandex-team.ru/infra/goxcart/pkg/unistats"
)

// Unistat is a Source which stores unistats response from other unistats server
type Unistat struct {
	URL string

	lastStats unistats.UniStats
	mu        sync.RWMutex
}

// Update gets new unistats from given s.URL. In case of error, last obtained unistats are dropped.
func (s *Unistat) Update(ctx context.Context) (err error) {
	defer func() {
		if err != nil {
			// drop in case of error
			s.setLastStats(nil)
		}
	}()
	req, err := http.NewRequestWithContext(ctx, http.MethodGet, s.URL, nil)
	if err != nil {
		return fmt.Errorf("unable to create request: %w", err)
	}

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return fmt.Errorf("unable to GET from %q: %w", s.URL, err)
	}
	defer resp.Body.Close()

	var tmp unistats.UniStats
	if err := json.NewDecoder(resp.Body).Decode(&tmp); err != nil {
		return fmt.Errorf("unable to parse json: %w", err)
	}
	s.setLastStats(tmp)
	return nil
}

func (s *Unistat) Drop() {
	s.setLastStats(unistats.UniStats{})
}

// setLastStats atomically sets last obtained unistats
func (s *Unistat) setLastStats(last unistats.UniStats) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	s.lastStats = last
}

func (s *Unistat) String() string {
	return fmt.Sprintf("balancer unistats from %q", s.URL)
}

func (s *Unistat) Unistats() unistats.UniStats {
	s.mu.Lock()
	defer s.mu.Unlock()

	return s.lastStats
}
