package watch

import (
	"context"
	"fmt"
	"log"
	"time"

	"golang.org/x/sync/errgroup"
)

// Updater can update itself
type Updater interface {
	// Stringer is used for logging errors
	fmt.Stringer

	// Update updates the updater.
	// It case of error, Drop may be called by the caller of Update(),
	// not inside Update itself.
	Update(ctx context.Context) error

	// Drop is used to drop stats to zero value
	// in case there was an error.
	Drop()
}

// UpdateEvery starts an infinite loop which updates given Updaters each time interval is passed.
func UpdateEvery(ctx context.Context, interval time.Duration, updaters []Updater, logger *log.Logger) error {
	ticker := time.NewTicker(interval)
	for {
		select {
		case <-ctx.Done():
			return ctx.Err()
		case <-ticker.C:
			updCtx, cancelCtx := context.WithTimeout(ctx, interval)
			if err := doUpdate(updCtx, updaters); err != nil {
				logger.Println(err)
			}
			cancelCtx()
		}
	}
}

func doUpdate(ctx context.Context, updaters []Updater) error {
	var g errgroup.Group
	for _, src := range updaters {
		src := src // https://golang.org/doc/faq#closures_and_goroutines
		g.Go(func() error {
			if err := src.Update(ctx); err != nil {
				src.Drop()
				return fmt.Errorf("unable to update %q: %w", src, err)
			}
			return nil
		})
	}
	return g.Wait()
}
