package watch

import (
	"context"
	"errors"
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type updateCounter int

func (u *updateCounter) String() string {
	return "test"
}

func (u *updateCounter) Update(_ context.Context) error {
	*u++
	return nil
}

func (u *updateCounter) Drop() {
	*u = 0
}

func Test_doUpdate(t *testing.T) {
	counts := []struct {
		updatersCount int
		callCount     int
	}{
		{
			updatersCount: 1,
			callCount:     1,
		},
		{
			updatersCount: 1,
			callCount:     5,
		},
		{
			updatersCount: 5,
			callCount:     1,
		},
		{
			updatersCount: 5,
			callCount:     10,
		},
		{
			updatersCount: 10,
			callCount:     5,
		},
	}
	for _, c := range counts {
		t.Run(fmt.Sprintf("%dx%d", c.updatersCount, c.callCount), func(t *testing.T) {
			var testUpdaters []*updateCounter
			for i := 0; i < c.updatersCount; i++ {
				testUpdaters = append(testUpdaters, new(updateCounter))
			}
			var updaters []Updater
			for _, u := range testUpdaters {
				updaters = append(updaters, u)
			}
			for i := 0; i < c.callCount; i++ {
				require.NoError(t, doUpdate(context.Background(), updaters))
			}
			for i, u := range testUpdaters {
				assert.EqualValuesf(t, c.callCount, *u, "updater#%d was updated %d times, but expected %d", i, u, c.callCount)
			}
		})
	}
}

var errUpdateFailed = errors.New("test update failed")

type testErrorUpdater struct {
	c     updateCounter
	count int
}

func (u *testErrorUpdater) String() string {
	return "testError"
}

func (u *testErrorUpdater) Update(ctx context.Context) error {
	_ = u.c.Update(ctx)
	u.count++
	return errUpdateFailed
}

func (u *testErrorUpdater) Drop() {
	u.count = 0
}

func Test_doUpdate_errors(t *testing.T) {
	counts := []struct {
		updatersCount int
		callCount     int
	}{
		{
			updatersCount: 1,
			callCount:     1,
		},
		{
			updatersCount: 1,
			callCount:     5,
		},
		{
			updatersCount: 5,
			callCount:     1,
		},
		{
			updatersCount: 5,
			callCount:     10,
		},
		{
			updatersCount: 10,
			callCount:     5,
		},
	}
	for _, c := range counts {
		t.Run(fmt.Sprintf("%dx%d", c.updatersCount, c.callCount), func(t *testing.T) {
			var testUpdaters []*testErrorUpdater
			for i := 0; i < c.updatersCount; i++ {
				testUpdaters = append(testUpdaters, new(testErrorUpdater))
			}
			var updaters []Updater
			for _, u := range testUpdaters {
				updaters = append(updaters, u)
			}
			for i := 0; i < c.callCount; i++ {
				require.Error(t, doUpdate(context.Background(), updaters))
			}
			for i, u := range testUpdaters {
				assert.Equal(t, 0, u.count, "updater%d did not drop its value")
				assert.EqualValuesf(t, c.callCount, u.c, "updater#%d was updated %d times, but expected %d", i, u.c, c.callCount)
			}
		})
	}
}
