import py
import os
import platform
from multiprocessing import cpu_count

from doit.tools import run_once

#DOIT_CONFIG = {
#    'default_tasks': ['toolchain:rich']
#}
SRCDIR = py.path.local(__file__).dirpath().join('src')
BUILDDIR = py.path.local('build').ensure(dir=1)
DLDIR = BUILDDIR.join('dl').ensure(dir=1)
PKGDIR = BUILDDIR.join('packages').ensure(dir=1)

SOURCES = {
    'binutils': [
        'ee0f10756c84979622b992a4a61ea3f5', '%(name)s-2.22.tar.bz2',
        'http://ftp.gnu.org/gnu/binutils/%(fn)s'
    ],
    #'gcc': [
    #    '933e6f15f51c031060af64a9e14149ff', '%(name)s-4.7.1.tar.bz2',
    #    'http://gcc.fyxm.net/releases/gcc-4.7.1/%(fn)s'
    #],
    'gcc': [
        '', '%(name)s-4.2.1.tar.bz2',
        'http://gcc.fyxm.net/releases/gcc-4.2.1/%(fn)s'
    ],
    'uClibc': [
        'a338aaffc56f0f5040e6d9fa8a12eda1', '%(name)s-0.9.33.2.tar.bz2',
        'http://uclibc.org/downloads/%(fn)s'
    ],
    'kernel': [
        '925bc1a6e2c98cb1e4d9ed8d30986dfb', 'linux-3.5.3.tar.bz2',
        'http://www.kernel.org/pub/linux/kernel/v3.0/%(fn)s'
    ],
    #'gmp': [
    #    '041487d25e9c230b0c42b106361055fe', '%(name)s-5.0.5.tar.bz2',
    #    'ftp://ftp.gmplib.org/pub/gmp-5.0.5/%(fn)s'
    #],
    #'mpfr': [
    #    'e90e0075bb1b5f626c6e31aaa9c64e3b', '%(name)s-3.1.1.tar.bz2',
    #    'http://www.mpfr.org/mpfr-current/%(fn)s'
    #],
    #'mpc': [
    #    '13370ceb2e266c5eeb2f7e78c24b7858', '%(name)s-1.0.tar.gz',
    #    'http://www.multiprecision.org/mpc/download/%(fn)s'
    #]
}

# Example 1: we are on x86_64-gnu and want to build cross compiler for i686-uclibc
# 1. simple cross, which can compile c code for i686-uclibc target on source host
# 2. rich cross using first cross, which can compile c and c++ code for i686-uclibc target on source host
# 3. (optional) native compiler which can compiler c and c++ code for i686-uclibc target on target host

THIS_PLATFORM = '%s-%s' % (platform.machine(), platform.system().lower())
TARGET_PLATFORM = '%s-%s-%s' % ('i686', 'linux', 'uclibc')

HOST = platform.machine() + '-linux-gnu'
BUILD = 'i686-cross-linux-uclibc'
TARGET = 'i686-linux-uclibc'
TOOLCHAIN_PREFIX = 'i686-'

CPUCOUNT = cpu_count()

CROSSDIR = BUILDDIR.join('snapshot/cross')
CROSSBUILDDIR = BUILDDIR.join('cross')
NATIVEDIR = BUILDDIR.join('snapshot/native')
NATIVEBUILDDIR = BUILDDIR.join('native')

HOSTDIR = BUILDDIR.join('build.host')
TINYDIR = BUILDDIR.join('build.tiny')
RICHDIR = BUILDDIR.join('build.rich')
HOSTSTAGEDIR = BUILDDIR.join('stage.host')
TINYSTAGEDIR = BUILDDIR.join('stage.tiny')
RICHSTAGEDIR = BUILDDIR.join('stage.rich')

CFLAGS = '-g -pipe -Os'
CXXFLAGS = CFLAGS
LDFLAGS = ''

MAKE = 'make -j%d' % (CPUCOUNT, )

# TODO: DROP
STAGE0 = BUILDDIR.join('stage0')
STAGE1 = BUILDDIR.join('stage1')

os.environ['CCACHE_DIR'] = BUILDDIR.join('.ccache').strpath


def _blank_path(p):
    p = py.path.local(p)
    if p.check(exists=1) or p.check(link=1):
        p.remove()
    p.ensure(dir=1)


def _run_shell(s):
    return ['set -e -x'] + s


def _use_toolchain(which, script, task_dep, ccache=True):
    assert which in ('system', 'cross', 'native')

    if which in ('cross', 'native'):
        add_path = CROSSDIR if which == 'cross' else NATIVEDIR

        task_dep.append('toolchain:' + which)
        script.append('export PATH="%s:$PATH"' % (add_path.join('bin')))

        #for prog in (
        #    'ar', 'as', 'cpp', 'ld', 'objcopy', 'objdump', 'ranlib', 'readelf', 'strip',
        #    'ld', 'nm',
        #    ('CXX', 'c++'),
        #    ('CC', 'cc')
        #):
        #    if isinstance(prog, tuple):
        #        export, name = prog
        #    else:
        #        export, name = prog.upper(), prog

        #    if which == 'cross':
        #        name = BUILD + '-' + name
        #    #script.append('echo "EXPORT %s=%s" >&2' % (export, name))
        #    script.append('export %s="%s"' % (export, name))

    if ccache:
        script.append('[ -z $CC ] && CC=cc; export CC="/usr/bin/ccache $CC"')
        script.append('[ -z $CXX ] && CXX=c++; export CXX="/usr/bin/ccache $CXX"')

    script.append('export CFLAGS="%s"' % (CFLAGS, ))
    script.append('export CXXFLAGS="%s"' % (CXXFLAGS, ))
    script.append('export LDFLAGS="%s"' % (LDFLAGS, ))


def _make_script(script):
    result = []
    for s in script:
        #result.append('echo "++ %s" >&2' % (s.replace('{', '{{').replace('}', '}}'), ))
        if isinstance(s, (list, tuple)):
            s = '\n'.join(s)
        result.append('echo "++ %s" >&2' % (s, ))
        result.append(s.replace('\n', ' \\\n'))

    return ';\n'.join(result)


def task_download_source():  # {{{
    for name, (checksum, fn, url) in SOURCES.items():
        fn = fn % {'name': name}
        url = url % {'name': name, 'fn': fn}
        rfn = DLDIR.join(fn)

        yield {
            'name': name.lower(),
            'actions': [
                'wget -c -O "%s" "%s"' % (
                    rfn.strpath, url
                )
            ],
            'targets': [rfn.strpath],
            'uptodate': [
                lambda task, values: \
                    rfn.check(exists=1) and (
                        checksum == '' or rfn.computehash() == checksum
                    )
            ]
        }
# }}}


def task_ccache_masquerade():
    def _make_links(system):
        links = {
            'cc': None,
            'c++': None,
            'gcc': None,
            'g++': None
        }

        if system == 'system':
            for name in links:
                links[name] = name
            BINDIR = HOSTSTAGEDIR.join('bin').ensure(dir=1)
        elif system == 'tiny':
            for name in links:
                links[name] = TARGET_PLATFORM + '-' + name
            BINDIR = TINYSTAGEDIR.join('bin/ccache').ensure(dir=1)

        for name, symlink in links.items():
            TGT = BINDIR.join(symlink)
            if TGT.check(link=1) or TGT.check(exists=1):
                TGT.remove()
            TGT.mksymlinkto('/usr/bin/ccache')

        paths = os.environ['PATH'].split(':')

        if system == 'system':
            if BINDIR.strpath in paths:
                paths.remove(BINDIR.strpath)

            paths.insert(0, BINDIR.strpath)
            os.environ['PATH'] = ':'.join(paths)

    for name in ('system', 'tiny'):
        yield {
            'name': name,
            'actions': [
                (_make_links, (name, ))
            ],
        }


def task_prepare_source():  # {{{
    for name, (_, fn, _) in SOURCES.items():
        fn = fn % {'name': name}
        target_dir = PKGDIR.join(name)

        def _prepare(target_dir):
            if target_dir.check(exists=1) or target_dir.check(link=1):
                target_dir.remove()
            target_dir.ensure(dir=1)

        rfn = DLDIR.join(fn)

        if fn.endswith('.tar.bz2') or fn.endswith('.tbz2'):
            unpack_cmd = 'tar --strip-components=1 -xjf "%s" -C "%s"' % (
                rfn.strpath, target_dir.strpath
            )
        elif fn.endswith('.tar.gz') or fn.endswith('.tgz'):
            unpack_cmd = 'tar --strip-components=1 -xzf "%s" -C "%s"' % (
                rfn.strpath, target_dir.strpath
            )
        else:
            unpack_cmd = 'exit 1'

        yield {
            'name': name.lower(),
            'actions': [
                (_prepare, (target_dir, )),
                unpack_cmd,
                _make_script([
                    'set -e',
                    'cd {target_dir}',
                    [
                        'for p in {SRCDIR}/patches/{name}*; do',
                        '    if [ -e $p ]; then',
                        '        patch -p0 -i $p;',
                        '    fi;',
                        'done',
                    ]
                ]).format(**dict(globals().items() + locals().items()))
            ],
            'file_dep': [rfn.strpath],
            'uptodate': [
                (lambda task, values, target_dir: target_dir.check(exists=1, dir=1), (target_dir, ))
            ]
        }
# }}}


def task_binutils():  # {{{
    def mk(which):
        actions = []
        task_dep = ['prepare_source:binutils']

        if which == 'tiny':
            CBUILDDIR = TINYDIR.join('binutils')
            PREFIX = TINYSTAGEDIR
            BUILD, HOST = None, None  # build using sys compiler, run on sys
            TARGET = TARGET_PLATFORM  # produced binutils should make stuff for our target platform
            task_dep.append('ccache_masquerade:system')
        else:
            CBUILDDIR = RICHDIR.join('binutils')
            PREFIX = RICHSTAGEDIR
            BUILD = THIS_PLATFORM
            HOST = TARGET_PLATFORM
            TARGET = TARGET_PLATFORM
            task_dep.append('ccache_masquerade:tiny')
            task_dep.append('toolchain:tiny')

        PREFIX, BUILD, HOST, TARGET  # mark not unused

        script = [
            'set -e',
            'cd {CBUILDDIR}'
        ]

        # Since we will use tiny compiler, we should add it to path
        if which == 'rich':
            script.append('export PATH={TINYSTAGEDIR}/bin/ccache:{TINYSTAGEDIR}/bin:$PATH')

        actions.append((_blank_path, (CBUILDDIR, )))

        if which == 'tiny':
            # This hack is required, becouse overwise binutils
            # produced will be for target platform.
            conf = [
                'AR=ar AS=as LD=ld NM=nm OBJDUMP=objdump OBJCOPY=objcopy'
            ]
        else:
            conf = [
                'AR={HOST}-ar CC={HOST}-cc'
            ]

        conf += [
            '{PKGDIR}/binutils/configure',
            '--prefix={PREFIX}',
            '--target={TARGET}',
            '--disable-nls',
            '--disable-shared',
            '--disable-multilib',
            '--disable-werror',
            '--with-lib-path=lib',
            '--program-prefix={TARGET}-'
        ]

        if which == 'rich':
            conf += [
                '--build={BUILD}',
                '--host={HOST}'
            ]

        #if targ == 'native':
        #    conf = [
        #        'AR={BUILD}-ar',
        #        'CC={BUILD}-cc',
        #    ] + conf

        #    conf += [
        #        '--build={BUILD}',
        #        '--host={HOST}'
        #    ]

        for idx, c in enumerate(conf):
            if idx > 0:
                conf[idx] = '    ' + c

        script += [
            'mkdir -p "{PREFIX}"',
            'cd {CBUILDDIR}',
            conf,
            '{MAKE} configure-host',
            '{MAKE} CFLAGS="--static %s"' % (CFLAGS, ),
            '{MAKE} install',
            'mkdir -p {PREFIX}/include',
            'cp {PKGDIR}/binutils/include/libiberty.h {PREFIX}/include/',
            'rm -rf {CBUILDDIR}'
        ]

        actions.append(_make_script(script).format(**dict(globals().items() + locals().items())))

        return {
            'name': which,
            'actions': actions,
            'task_dep': task_dep,
            'uptodate': [run_once]
        }

    for name in ('tiny', 'rich'):
        yield mk(name)
# }}}


def task_gcc():  # {{{
    def mk(which):
        actions = []
        task_dep = [
            'prepare_source:gcc',
            'binutils:' + which
        ]

        if which == 'tiny':
            CBUILDDIR = TINYDIR.join('gcc')
            PREFIX = TINYSTAGEDIR
            BUILD = None  # build using sys compiler, run on sys
            HOST = THIS_PLATFORM      # produced gcc should run here
            TARGET = TARGET_PLATFORM  # produced gcc should make stuff for our target platform
            CCWRAP_CC = 'gcc'
            task_dep.append('ccache_masquerade:system')
        else:
            CBUILDDIR = RICHDIR.join('gcc')
            PREFIX = RICHSTAGEDIR
            BUILD = THIS_PLATFORM
            HOST = TARGET_PLATFORM
            TARGET = TARGET_PLATFORM
            CCWRAP_CC = TARGET_PLATFORM + '-cc'
            task_dep.append('ccache_masquerade:tiny')
            task_dep.append('uclibc:rich')

        BUILD, HOST, TARGET, CCWRAP_CC  # dont mark as unused

        script = [
            'set -e',
            'cd {CBUILDDIR}',
            'export PATH={TINYSTAGEDIR}/bin/ccache:{TINYSTAGEDIR}/bin:$PATH',
        ]

        actions.extend([
            (_blank_path, (CBUILDDIR, )),
            (_blank_path, (PREFIX.join('cc'), )),
            (_blank_path, (PREFIX.join('cc/lib'), )),
        ])

        actions.append((_blank_path, (CBUILDDIR, )))

        conf = [
            '{PKGDIR}/gcc/configure',
            '--prefix={PREFIX}',
            '--target={TARGET}',
            '--disable-nls',
            '--disable-multilib',
            '--enable-c99',
            '--enable-long-long',
            '--enable-__cxa_atexit',
        ]

        if which == 'tiny':
            conf += [
                '--enable-languages="c"',
                '--disable-threads',
                '--disable-shared',
                '--host={HOST}',
            ]
        else:
            conf = [
                'CC={TARGET}-cc',
                'AR={TARGET}-ar',
                'AS={TARGET}-as',
                'LD={TARGET}-ld',
                'NM={TARGET}-nm',
                'CC_FOR_TARGET={TARGET}-cc',
                'AR_FOR_TARGET={TARGET}-ar',
                'NM_FOR_TARGET={TARGET}-nm',
                'AS_FOR_TARGET={TARGET}-as',
                'LD_FOR_TARGET={TARGET}-ld',
                'GCC_FOR_TARGET={TARGET}-cc',
                'CXX_FOR_TARGET={TARGET}-c++',
                'ac_cv_path_AR_FOR_TARGET={TARGET}-ar',
                'ac_cv_path_RANLIB_FOR_TARGET={TARGET}-ranlib',
                'ac_cv_path_NM_FOR_TARGET={TARGET}-nm',
                'ac_cv_path_AS_FOR_TARGET={TARGET}-as',
                'ac_cv_path_LD_FOR_TARGET={TARGET}-ld',
            ] + conf

            conf += [
                '--enable-languages="c,c++"',
                '--enable-threads=posix',
                '--enable-shared',
                '--disable-libstdcxx-pch',
                '--build={BUILD}',
                '--host={HOST}',
            ]

        for idx, c in enumerate(conf):
            if idx > 0:
                conf[idx] = '    ' + c

        #if targ == 'native':
        #    xgcc_symlink = [
        #        'mkdir -p gcc',
        #        'ln -sf {CROSSDIR}/bin/{BUILD}-cc gcc/xgcc',
        #    ]
        #else:
        #    xgcc_symlink = []

        script += [
            'sed -i "s@^STMP_FIX.*@@" "{PKGDIR}/gcc/Makefile.in"',
            'cd {CBUILDDIR}',

            conf,

        #] + xgcc_symlink + [

            '{MAKE} configure-host',
            '{MAKE} all-gcc LDFLAGS="--static"',

            # GCC bug: we disabled multilib, but it doesn't always notice that
            'ln -sf lib {PREFIX}/lib64',
            '{MAKE} install-gcc',
            'rm {PREFIX}/lib64',

            # Move gcc internal lib somewhere sane
            'rm -rf {PREFIX}/lib/gcc/*/*/install-tools',
            'mv {PREFIX}/lib/gcc/*/*/include {PREFIX}/cc/include',
            'mv {PREFIX}/lib/gcc/*/*/* {PREFIX}/cc/lib',

            # Move compiler internal libs to "tools"
            'ln -sf {TARGET} {PREFIX}/tools',
            'cp {PREFIX}/libexec/gcc/*/*/c* {PREFIX}/tools/bin',

            # Prepare for ccwrap
            'mv {PREFIX}/bin/{TARGET}-gcc {PREFIX}/tools/bin/cc',
            'ln -sf {TARGET}-cc {PREFIX}/bin/{TARGET}-gcc',
            'ln -sf cc {PREFIX}/tools/bin/rawcc',

            # Make ccwrap
            'cp {SRCDIR}/ccwrap.c .',
            'CFLAGS="%s" {CCWRAP_CC} ccwrap.c --static -o {PREFIX}/bin/{TARGET}-cc' % (CFLAGS, ),
            'echo -e \'#!/bin/sh\\n\\nexec {TARGET}-cc -E "$@"\' > {PREFIX}/bin/{TARGET}-cpp',
            'chmod +x {PREFIX}/bin/{TARGET}-cpp',

            # Cleanup
            'rm -rf {PREFIX}/libexec',
            'rm -rf {PREFIX}/lib/gcc',
            'rm -rf {PREFIX}/bin/g*',
            'rm -rf {PREFIX}/bin/c*',
            'rm -rf {CBUILDDIR}',
        ]

        actions.append(_make_script(script).format(**dict(globals().items() + locals().items())))

        return {
            'name': which,
            'actions': actions,
            'task_dep': task_dep,
            'uptodate': [run_once]
        }

    for name in ('tiny', 'rich'):
        yield mk(name)
# }}}


def task_uclibc():  # {{{
    def mk(which):
        actions = []
        task_dep = [
            'prepare_source:uclibc',
            'linux_headers:' + which,
            'ccache_masquerade:tiny',
        ]

        CROSS = TARGET_PLATFORM + '-'

        if which == 'tiny':
            PREFIX = TINYSTAGEDIR
        else:
            PREFIX = RICHSTAGEDIR
            task_dep.append('toolchain:tiny')

        CROSS, PREFIX  # mark not unused

        task_dep.append('gcc:tiny')

        script = [
            'set -e',
            'export PATH={TINYSTAGEDIR}/bin/ccache:{TINYSTAGEDIR}/bin:$PATH',
            'cd {PKGDIR}/uClibc',
            'cp {SRCDIR}/config-uClibc .config',
            '{MAKE} oldconfig',
            'mkdir -p {PREFIX}/src {PREFIX}/bin',
            'cp .config {PREFIX}/src/config-uClibc',
            [
                '{MAKE}',
                '    CROSS={CROSS}',
                '    UCLIBC_LDSO_NAME=ld-uClibc',
                '    KERNEL_HEADERS={PREFIX}/include',
                '    PREFIX={PREFIX}',
                '    RUNTIME_PREFIX=/',
                '    DEVEL_PREFIX=/',
                '    install'
            ],
        ]

        if which == 'rich':
            script += [
                [
                    '{TARGET_PLATFORM}-cc utils/ldd.c -o {PREFIX}/bin/{TARGET_PLATFORM}-ldd',
                    '--static -I ldso/include -DBUILDING_LINKAGE -DUCLIBC_LDSO=\'"ld-uClibc.so.0"\''
                ],
                [
                    '{TARGET_PLATFORM}-cc utils/ldconfig.c utils/chroot_realpath.c',
                    '-o {PREFIX}/bin/{TARGET_PLATFORM}-ldconfig',
                    '--static -I ldso/include -DUCLIBC_RUNTIME_PREFIX=\'"/"\' -DBUILDING_LINKAGE'
                ]
            ]

        actions.append(_make_script(script).format(**dict(globals().items() + locals().items())))

        return {
            'name': which,
            'actions': actions,
            'task_dep': task_dep,
            'uptodate': [run_once]
        }

    for name in ('tiny', 'rich'):
        yield mk(name)
# }}}


def task_linux_headers():  # {{{
    for name, PREFIX in (
        ('tiny', TINYSTAGEDIR),
        ('rich', RICHSTAGEDIR)
    ):

        script = [
            'set -e',
            'cd {PKGDIR}/kernel',
            '{MAKE} distclean',
            '{MAKE} ARCH=i386 defconfig',
            '{MAKE} headers_install ARCH=i386 INSTALL_HDR_PATH={PREFIX}',
            'ln -sf ../sys/user.h {PREFIX}/include/asm/page.h',
            'find {PREFIX}/include -name ".install" -delete',
            'find {PREFIX}/include -name "..install.cmd" -delete',
            '{MAKE} distclean',
        ]

        yield {
            'name': name,
            'actions': [
                _make_script(script).format(**dict(globals().items() + locals().items()))
            ],
            'task_dep': ['prepare_source:kernel'],
            'uptodate': [run_once],
        }
# }}}


def task_toolchain():
    yield {
        'name': 'tiny',
        'actions': [],
        'task_dep': [
            'gcc:tiny',
            'uclibc:tiny'
        ]
    }

    yield {
        'name': 'rich',
        'actions': [
            'set -e -x;'
            'tar -C {RICHSTAGEDIR} -cjf {TARGET_PLATFORM}.tbz2 bin cc {TARGET_PLATFORM} include lib src tools'
            .format(**globals())
        ],
        'task_dep': [
            'gcc:rich',
            'uclibc:rich'
        ]
    }
