#!/usr/bin/env bash

DEPENDENCIES="api kernel library"

set -e

require_python()
{
    if [ -z "${PYTHON}" ]; then
        echo "Please export \$PYTHON variable!"
        exit 1
    else
        true
    fi
}

do_info()
{
    echo " * " $*
}

do_cleanup()
{
    do_info "cleanup $*"

    rm -rf ./dist ./*.egg-info

    if [ "$1" = "full" ]; then
        rm -rf ./build ./snapshot
    fi
}

do_bin_snapshot()
{
    do_info "adding binary files"
    for f in $(ls ./bin); do
        if [ "$1" = "symlink" ]; then
            ln -s ../../bin/$f ./snapshot/bin
        elif [ "$1" = "link" ]; then
            cp --link ./bin/$f ./snapshot/bin
        else
            cp ./bin/$f ./snapshot/bin
        fi
    done
}

do_prepare()
{
    do_info "preparing directories"
    mkdir -p ./snapshot/lib ./snapshot/bin
}

do_ctl()
{
    do_info "adding heartbeat-client.scsd"

    if [ "$1" = "symlink" ]; then
        ln -sf ../heartbeat-client.scsd snapshot
    else
        cp -f ./heartbeat-client.scsd snapshot
    fi

    $PYTHON -m compileall -q snapshot
}

do_call_setuppy()
{
    do_info "running: setup.py $*"
    PYTHONPATH=$PYTHONPATH:./snapshot/lib $PYTHON setup.py -q "$@"
}

do_share()
{
    do_info "adding share"
    rm -rf snapshot/share

    if [ x$1 = xsymlink ]; then
        ln -sf ../share snapshot/share
    else
        cp -fr share snapshot/share
    fi

    $PYTHON -m compileall -q snapshot
}

do_bundle()
{
    require_python
    do_cleanup full
    do_prepare
    do_ctl
    do_share
    do_call_setuppy egg_info bdist_egg
    do_call_setuppy easy_install -d ./snapshot/lib -s ./snapshot/bin -Z -O0 -m dist/*.egg

    do_info "update shebangs"
    for fn in $(find ./snapshot/bin -type f); do
        shebang=$(head -n1 ${fn})
        echo ${shebang} | grep -E '^#!' >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            if [ $(uname) = "Linux" ] || [[ $(uname) = "CYGWIN"* ]]; then
                sed -E -i -e '1s:.*:#!/skynet/python/bin/python:' ${fn}
            else
                sed -E -i '' -e '1s:.*:#!/skynet/python/bin/python:' ${fn}
            fi
        fi
    done

    mkdir -p snapshot/bin/plugins
    for path in $(ls ./snapshot/bin); do
        cut=${path##plugin__}
        if [ "$path" != "$cut" ]; then
            mv snapshot/bin/$path snapshot/bin/plugins/$cut
        fi
    done

    base=""
    libname=$(ls snapshot/lib | head -n1)

    for path in ya skynet services heartbeatclient; do
        if [ "$base" != "" ]; then
            path="$base/$path"
        fi
        base=$path

        if [ ! -f snapshot/lib/$libname/$path/__init__.py ]; then
            echo "Creating __init__.py for $path"
            echo "__import__(\"pkg_resources\").declare_namespace(__name__)" >  snapshot/lib/$libname/$path/__init__.py
        fi
    done

    do_cleanup
}

do_develop()
{
    require_python
    do_cleanup
    rm -rf ./snapshot

    do_prepare
    do_ctl symlink
    do_share symlink
    do_call_setuppy develop -d ./snapshot/lib -s ./snapshot/bin -z -O2 -m

    rm -rf snapshot/ve
    for mod in ya ya/skynet ya/skynet/services; do
        mkdir -p snapshot/ve/${mod}
        echo '__import__("pkg_resources").declare_namespace(__name__)' > snapshot/ve/${mod}/__init__.py
    done

    local _egg_info=$(ls -d *.egg-info)
    mv ${_egg_info} snapshot/ve

    local _egg_link=$(basename $(ls snapshot/lib/*.egg-link))
    echo "$(pwd)/snapshot/ve" > snapshot/lib/${_egg_link}
    echo "../.." >> snapshot/lib/${_egg_link}

    ln -s ../../../../../src ./snapshot/ve/ya/skynet/services/heartbeatclient
    do_cleanup
}

do_pylint()
{
    require_python
    PYTHONPATH=./snapshot/lib:$PYTHONPATH $PYTHON $(dirname $PYTHON)/pylint --rcfile=./pylint.conf ya.skynet.services.heartbeatclient "$@"
}

do_shell()
{
    require_python
    do_prepare
    cat > snapshot/shell <<END
#!/usr/bin/env python
from pkg_resources import require, load_entry_point
require('skynet-heartbeat-client-service')
raise SystemExit(load_entry_point('ipython', 'console_scripts', 'ipython')())
END
    chmod +x snapshot/shell
    PYTHONPATH=./snapshot/lib:$PYTHONPATH $PYTHON snapshot/shell
}

do_run()
{
    ./snapshot/bin/daemon -C AppPath=$(pwd)/snapshot "$@"
}

do_run_plugin()
{
    plugin=$1
    shift 1
    ./snapshot/bin/plugins/$plugin "$@"
}

do_tgz()
{
    do_bundle
    rm -rf heartbeat-client.tgz
    tar --exclude '.svn' -C snapshot -czvf heartbeat-client.tgz .
}

case "$1" in
    bundl*)
        do_bundle
        ;;

    dev*)
        do_develop
        ;;

    clean*)
        do_cleanup full
        ;;

    pylint*)
        shift
        do_pylint "$@"
        ;;

    shell*)
        do_shell
        ;;

    runplug*)
        shift 1
        do_run_plugin "$@"
        ;;

    run*)
        shift 1
        do_run "$@"
        ;;

    tgz*)
        do_tgz
        ;;

    dep*)
        echo ${DEPENDENCIES}
        ;;

    *)
        echo "Usage $0 [options] develop | bundle | clean | pylint | shell | run | tgz | deps | help"
        ;;
esac
