from __future__ import absolute_import, print_function, division

import re
from library import config

try:
    from api.srvmngr import getRoot
except ImportError:
    def getRoot():
        raise Exception("No srvmngr and do not know where is supervisor")

import six


REP_RE = re.compile('\$\{([a-zA-Z0-9_\-\.]+)\}')


class AppConfigSubConfigProxy(object):
    def __init__(self, config, key):
        self.config = config
        self.key = key

    def __getattr__(self, attr):
        return self.config.__getattr__('.'.join((self.key, attr)))


class AppConfig(dict):
    __proxyClass = AppConfigSubConfigProxy

    def __init__(self):
        self.__needProxy = {}

    def _flattenData(self, data):
        flattenData = {}
        left = [(None, k, v) for k, v in list(data.items())]

        while left:
            prefix, key, value = left.pop(0)

            if prefix is not None:
                key = '.'.join((prefix, key))

            if isinstance(value, dict):
                if not value.pop('_dict', 0):
                    for subKey, subValue in list(value.items()):
                        left.append((key, subKey, subValue))
                    continue

            flattenData[key] = value

        return flattenData

    def load(self, data):
        self.update(self._flattenData(data))

        toParse = list(self.keys())

        while toParse:
            key = toParse.pop(0)
            value = self[key]

            if '.' in key:
                tail = None
                for sKey in key.split('.')[:-1]:
                    if tail is not None:
                        sKey = '.'.join((tail, sKey))
                    self.__needProxy.setdefault(sKey, None)
                    tail = sKey

            if not isinstance(value, six.string_types) or '$' not in value:
                continue

            while True:
                match = REP_RE.search(value)
                if not match:
                    break

                replKey = match.group(1)

                # If replacement didnt parsed yet, continue
                if replKey in toParse:
                    toParse.append(key)
                    break

                start, end = match.span()

                if start == 0 and end == len(value) and not isinstance(self[replKey], six.string_types):
                    value = self[replKey]
                    break
                else:
                    value = ''.join((value[0:start], self[replKey], value[end:]))

            self[key] = value

    def __getattr__(self, attr):
        if attr in self.__needProxy:
            if self.__needProxy[attr] is None:
                self.__needProxy[attr] = self.__proxyClass(self, attr)
            return self.__needProxy[attr]

        if attr in self:
            return self[attr]
        raise AttributeError('No such config attribute %r' % attr)


def loadConfig(overrides={}):
    if 'SupervisorPath' not in overrides:
        overrides['SupervisorPath'] = getRoot()
    return config.query('skynet.services.heartbeat-client', overrides=overrides)
