"""
The purpose of this module is to make all needed stuff for starting daemon.
This includes initialization of our Context with DaemonContext inside.

During context initialization we will close file descriptors, setup
signal handlers and so on.

It is important what before context initialization gevent will NOT be imported.
"""

from __future__ import absolute_import, print_function, division

import os
import sys
import argparse

from kernel.util.errors import formatException

from .context import Context
from ..errors import CopierNGError


def parseArgs():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: \
                        argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description='Run copierng daemon.'
    )
    parser.add_argument('-d', '--daemonize', action='store_true')
    parser.add_argument('-b', '--background', action='store_true')
    parser.add_argument(
        '-v', '--verbose', metavar='N', nargs='?', const=1, type=int, default=0,
        help='Increase verbosity by N times.'
    )
    parser.add_argument(
        '-C', action='append', metavar='KEY=VALUE', default=[],
        help='Override config (can be specified several times)'
    )
    parser.add_argument('--trace-config', action='store_true', help='Trace configuration to stdout.')
    parser.add_argument('-l', '--list-reports', action='store_true', help='List all known reports.')
    parser.add_argument('-r', '--report', metavar='N', type=int, help='Show details of the report with ID specified.')

    return parser.parse_args()


def main():
    args = parseArgs()
    if args.list_reports or args.report:
        # Console-only command, do not start the daemon.
        from .. import main as realMain
        from ..api import server
        from . import config

        hb = realMain.HeartBeat
        cfg = config.loadConfig(overrides=dict(x.split('=', 1) for x in args.C))
        rpc = server(cfg=cfg.rpc, appdir=cfg.AppPath, rundir=cfg.WorkdirPath)
        return hb.listReports(rpc) if args.list_reports else hb.reportDetails(rpc, args.report, args.verbose)

    try:
        ctx = Context()
        ctx.initialize(args)
    except SystemExit:
        raise
    except CopierNGError as ex:
        print(
            'ERROR (initializing): {0}'.format(ex),
            file=sys.stderr
        )
        os._exit(1)

    except BaseException:
        print(
            'ERROR (initializing): {0}'.format(formatException()),
            file=sys.stderr
        )
        os._exit(1)  # pylint: disable=W0212

    try:
        with ctx:
            from .. import main as realMain
            return realMain.main(ctx)
    except SystemExit:
        raise
    except BaseException:
        ctx.log.critical('Unhandled error: %s' % (formatException(), ))
        os._exit(1)  # pylint: disable=W0212


if __name__ == '__main__':
    raise SystemExit(main())
