import re

from kernel.util import logging


def resolveInConfig(ctx, value):
    matches = re.finditer('\$\{(.*?)\}', value)

    if not matches:
        return value

    for match in matches:
        cfgValue = ctx.cfg
        for sub in match.group(1).split('.'):
            cfgValue = getattr(cfgValue, sub)

        value = value[:match.start()] + cfgValue + value[match.end():]

    return value


def findObject(name):
    if hasattr(logging, name):
        return getattr(logging, name)

    if '.' not in name:
        raise Exception('Cant find object {0!r}'.format(name))

    module, className = name.rsplit('.', 1)
    return getattr(__import__(module, fromlist=['']), className)


def createFormatters(conf):
    formatters = {}
    for name, props in conf.items():
        if '_class' in props:
            clsMod, clsName = props.pop('_class').rsplit('.', 1)
            mod = __import__(clsMod, fromlist=[clsName])
            cls = getattr(mod, clsName)
        else:
            cls = logging.Formatter      # pylint: disable=E1101
        formatters[name] = cls(**props)  # pylint: disable=W0142
    return formatters


def createLoggers(conf, handlers):
    loggers = {}
    for name, props in conf.items():
        logger = logging.getLogger(name)  # pylint: disable=E1101
        for handlerName in props.get('handlers', ()):
            logger.addHandler(handlers[handlerName])
        if 'level' in props:
            logger.setLevel(getattr(logging, props['level']))

        loggers[name] = logger
    return loggers


def createHandlers(ctx, conf, formatters):
    handlers = {}
    for name, props in conf.items():
        handlerClassName = props['class']

        handlerClass = findObject(handlerClassName)

        if 'args' in props:
            args = []
            for arg in props['args']:
                args.append(findObject(arg))
        else:
            args = []

        kwargs = props.pop('kwargs', {})
        for key, value in kwargs.items():
            kwargs[key] = resolveInConfig(ctx, value)

        handler = handlerClass(*args, **kwargs)  # pylint: disable=W0142

        if 'level' in props:
            handler.setLevel(getattr(logging, props['level']))

        if 'formatter' in props:
            formatter = formatters[props['formatter']]
            handler.setFormatter(formatter)

        handlers[name] = handler
    return handlers


def setupLogging(ctx):
    config = ctx.cfg
    logConf = config.logging

    logging.renameLevels(logConf.get('levelNameScheme', None))

    formatters = createFormatters(logConf['formatters'])
    handlers = createHandlers(ctx, logConf['handlers'], formatters)

    createLoggers(logConf['loggers'], handlers)
