# vim: set foldmethod=marker

from __future__ import absolute_import, print_function, division

import os
import bz2
import zlib
import time
import pprint

import msgpack
import gevent
import gevent.event
import gevent.queue

from kernel import util
from kernel.util.console import setProcTitle
from kernel.util.errors import formatException

from . import db
from .rpc.server import RPC
from .server import Server
from .plugins import PluginManager
from .reporter import Reporter
from .utils.greendeblock import Deblock


class HeartBeat(object):
    # Initialization {{{
    def __init__(self, ctx):
        self.ctx = ctx
        self.log = ctx.log.getChild('heartbeat')
        self.log.debug('Initializing')

        self._stopFlag = gevent.event.Event()
        self._startFlag = gevent.event.Event()

        self._reportQueue = gevent.queue.Queue()

        dbobj = db.Database(ctx)

        self._db_deblock = Deblock(keepalive=None, logger=self.log.getChild('deblock'))
        self._db = self._db_deblock.make_proxy(dbobj, put_deblock='deblock')
        self._db.open()
        self._db.init()

        self._workerGrn = None
        self._asyncReporterGrn = None

        self._server = None
        self._rpc = None
        self._pluginManager = None
        self._reporter = Reporter(self.ctx, self._db)

        self._svnInfo = None
        self._hostInfo = None
    # Initialization }}}

    # Management (start, stop, join) {{{
    def start(self):
        assert not self._stopFlag.isSet()

        self._server = Server(self.ctx).start()
        self._rpc = RPC(self.ctx).start()

        # Register server connection handlers
        self._server.registerConnectionHandler(self._rpc.getConnectionHandler())

        # Register RPC callbacks
        self._rpc.registerHandler('ping', self._ping)
        self._rpc.registerHandler('stop', self._stop)
        self._rpc.registerHandler('queueData', self._queueData)
        self._rpc.registerHandler('scheduleReport', self._scheduleReport)
        self._rpc.registerHandler('scheduleReportAsync', self._scheduleReportAsync)
        self._rpc.registerHandler('get_plugins', self._get_plugins)
        self._rpc.registerHandler('get_last_report', self._get_last_report)

        self._workerGrn = gevent.spawn(self._workerLoop)
        self._asyncReporterGrn = gevent.spawn(self._asyncReporterLoop)
        self._pluginManager = PluginManager(self.ctx, self._reporter.report, self._reporter.schedulerState()).start()

        self._reporter.start()
        self._stopFlag.clear()
        self._startFlag.set()
        return self

    def stop(self):
        assert self._startFlag.isSet()

        self._workerGrn.kill(gevent.GreenletExit)
        self._workerGrn.join()

        self._asyncReporterGrn.kill(gevent.GreenletExit)
        self._asyncReporterGrn.join()

        self._reporter.stop().join()
        self._pluginManager.stop().join()
        self._server.stop().join()
        self._rpc.stop().join()
        self._stopFlag.set()
        self._startFlag.clear()
        return self

    def join(self):
        self._stopFlag.wait()
    # Management (start, stop, join) }}}

    # RPC Meths {{{
    @RPC.full
    def _ping(job, self):
        if self._startFlag.wait(timeout=30):
            return True
        job.log.warning('Ping timed out waiting startFlag to become setted (wait 30 seconds)')
        return False

    @RPC.simple
    def _stop(self):
        self._stopFlag.set()

    @RPC.simple
    def _scheduleReport(self, name, report, valid, compress, incremental, discards, sendDelay):
        now = time.time()
        data = self._pluginManager.compress(
            {
                'name': name,
                'start': now,
                'end': now,
                'valid': valid,
                'report': report,
                'discards': discards,
                'incremental': incremental,
            },
            compression=('bz2' if compress else None) if not isinstance(compress, basestring) else compress
        )

        self._reporter.report(data, sendDelay)
        return True

    @RPC.simple
    def _scheduleReportAsync(self, name, report, valid, compress, incremental, discards, sendDelay):
        now = time.time()
        data = self._pluginManager.compress(
            {
                'name': name,
                'start': now,
                'end': now,
                'valid': valid,
                'report': report,
                'discards': discards,
                'incremental': incremental,
            },
            compression=('bz2' if compress else None) if not isinstance(compress, basestring) else compress
        )

        self._reportQueue.put((data, sendDelay))
        return True

    @RPC.full
    def _get_plugins(job, self):
        plugins = self._pluginManager.get_plugins()
        return plugins

    @RPC.full
    def _get_last_report(job, self, plugin_name):
        report = self._reporter.plugin_report(plugin_name)
        job.log.info('{} last report: {}'.format(plugin_name, report))
        return report

    @RPC.simple
    def _queueData(self, id_=None):
        return self._reporter.list() if id_ is None else self._reporter.details(id_)
    # RPC Meths }}}

    def _workerLoop(self):
        log = self.log.getChild('worker')
        log.info('Started')

        while 1:
            try:
                if self.ctx.args.daemonize:
                    ppid = os.getppid()
                    if ppid == 1:
                        log.warning('Parent PID in daemonized mode is %r, exiting!', ppid)
                        os.kill(os.getpid(), 1)  # Send SIGHUP signal to ourselves

                gevent.sleep(1)

            except gevent.GreenletExit:
                log.info('Received stop signal')
                break

            except Exception:
                log.error('Unhandled exception: %s', formatException())
                gevent.sleep(0.2)  # avoid busy loops

    def _asyncReporterLoop(self):
        log = self.log.getChild('asyncreporter')
        log.info('Started')

        last_report = time.time()
        while 1:
            try:
                # wait on queue
                data, sendDelay = self._reportQueue.get()
                self._reporter.report(data, sendDelay)

                if (time.time() - last_report) > 60:
                    log.info('Queue size %d', self._reportQueue.qsize())
                    last_report = time.time()

            except gevent.GreenletExit:
                log.info('Received stop signal')
                break

            except Exception:
                log.error('Unhandled exception: %s', formatException())
                gevent.sleep(0.2)  # avoid busy loops

    @staticmethod
    def listReports(rpc):
        sent2str = {0: 'Not sent', 1: 'Sending', 2: 'Sent'}

        list_ = [('ID', 'Type', 'Sent Date', 'Expiration Date', 'Send Attempts', 'Sent Status'), ]
        list_ += map(
            lambda r: (str(r[0]), r[1], util.ts2str(r[2]), util.ts2str(r[3]), str(r[4]), sent2str[r[5]], ),
            rpc.call('queueData')
        )

        lens = [max(map(lambda x: len(x[i]) + 1, list_)) for i in range(len(list_[0]))]
        list_.insert(1, tuple(map(lambda x: '-' * x, lens)))
        for line, row in enumerate(list_):
            for i, x in enumerate(row):
                print("|{0: <{len}}".format(x, len=lens[i]), end='')
            print('|')
        return 0

    @staticmethod
    def reportDetails(rpc, id_, verbose=0):
        sent2str = {0: 'Not sent', 1: 'Sending', 2: 'Sent'}
        unpack = {None: lambda x: x, 'zip': lambda x: zlib.decompress(x), 'bz2': lambda x: bz2.decompress(x)}
        decode = {
            None: lambda x: x if not isinstance(x, str) else msgpack.unpackb(x),
            'msgpack': lambda x: msgpack.unpackb(x)
        }

        r = rpc.call('queueData', id_)
        if not r:
            print('No such report #%d' % id_)
            return 1
        meta = [
            "ID", "Type", "Date Added", "Actualization Date", "Sent Date", "Expiration Date", "Format",
            "Compressed Size", "Decompressed Size", "Compression", "SHA1 Checksum", "Send Attempts", "Sent Status",
        ]
        maxLen = max([len(m) for m in meta])
        data = unpack[r[8]](r[7])
        r = \
            list(r[0:2]) + \
            map(util.ts2str, r[2:6]) + \
            [r[6], util.size2str(len(r[7])), util.size2str(len(data))] + \
            list(r[8:11]) + \
            [sent2str[r[11]]]
        if verbose:
            meta.append('Data')
            r.append(str(decode[r[6]](data)) if verbose < 2 else pprint.pformat(decode[r[6]](data)))
        for k, v in zip(meta, r):
            print('{0: <{len}}: {1:}'.format(k, v, len=maxLen))
        return 0


def main(ctx):
    log = ctx.log
    # Initialize daemon context
    setProcTitle(ctx.cfg.ProgramName)
    log.info('Initializing {0}'.format(ctx.cfg.ProgramName))

    app = HeartBeat(ctx).start()
    try:
        app.join()
    except KeyboardInterrupt:
        log.info('Caught SIGINT, stopping daemon')
    except SystemExit as ex:
        if ex.args:
            # We expect exception if this case will have meaningfull message
            log.warning(ex)
        else:
            log.warning('Got SystemExit exception in main loop')
        raise
    except BaseException:
        log.critical('Unhandled exception: %s, exit immidiately!' % formatException())
        os._exit(1)  # pylint: disable=W0212

    app.stop()

    with gevent.Timeout(2) as timeout:
        try:
            app.join()
        except gevent.Timeout as err:
            if err != timeout:
                raise
            log.error('Failed to stop {0!r} gracefully -- timeout occured.'.format(app))
            os._exit(1)  # pylint: disable=W0212

    log.info('Stopped {0}'.format(ctx.cfg.ProgramName))
    os._exit(0)
    return 0
