import os
import py
import bz2
import zlib
import yaml
import time
import random
import bisect
import msgpack
import traceback

import gevent

from kernel import util
from kernel.util.errors import formatException

from library import config as libconfig

from ..daemon.config import AppConfig
from ..utils import greensubprocess as subproc


# Exceptions {{{
class PluginRunError(Exception):
    pass


class PluginRunTimedOut(PluginRunError):
    pass


class PluginReportTooBig(PluginRunError):
    def __init__(self, size, limit):
        super(PluginReportTooBig, self).__init__()
        self.size = size
        self.limit = limit

    def __str__(self):
        return \
            'Generated report size %s exceed maximum allowed %s.' % \
            (util.size2str(self.size), util.size2str(self.limit))

# Exceptions }}}


class PluginManager(object):  # {{{
    CHECK_GLOBAL_CONFIG_INTERVAL = 300
    CHECK_FILES_INTERVAL = 60

    # Initialization {{{
    def __init__(self, ctx, report_func, state):
        self.ctx = ctx
        self.cfg = ctx.cfg.plugin_manager
        self.log = ctx.log.getChild('plugins')
        self.log.debug('Initializing')

        self._plugins = {}              # name => (plugin, conf)
        self._plugins_from_file = {}    # file => (name, conf_mtime)

        self._global_config_check_time = 0
        self._files_check_time = 0

        self._state = state             # name => state
        self._pluginRunQueue = []
        self._workerGrn = None
        self._report = report_func

        for path in (
            self.cfg.default_plugins.configs,
            self.cfg.default_plugins.binaries,
        ):
            try:
                py.path.local(path).ensure(dir=1)
            except py.error.EACCES:
                pass
    # Initialization }}}

    # Management (start, stop, join) {{{
    def start(self):
        assert self._workerGrn is None
        self._workerGrn = gevent.spawn(self._worker_loop)
        return self

    def stop(self):
        assert self._workerGrn is not None
        self._workerGrn.kill(gevent.GreenletExit)
        return self

    def join(self):
        self._workerGrn.join()
    # Management (start, stop, join) }}}

    def get_plugins(self):  # {{{
        return self._plugins.keys()
    # }}}

    # Compress {{{
    @staticmethod
    def compress(report, compression='bz2'):
        data = msgpack.dumps(report['report'])
        compressor = {
            None: lambda x: x,
            'bz2': bz2.compress,
            'zip': zlib.compress,
        }[compression]
        report.update({
            'report': compressor(data),
            'compression': compression,
            'format': 'msgpack',
        })
        return report
    # }}}

    def _run_plugin(self, plugin):  # {{{
        now = time.time()
        log = self.log.getChild('worker')

        try:
            if plugin.next_run_timestamp > now:
                log.warn(
                    'The plugin run requested %s early than the planned time. Skip this run.',
                    util.td2str(now - plugin.next_run_timestamp)
                )
                return

            if plugin.prev_run_timestamp is not None:
                log.info('Running plugin: %s, prev run was %s ago' % (
                    plugin, util.td2str(now - plugin.prev_run_timestamp)
                ))
            else:
                log.info('Running plugin: %s, first time' % (plugin, ))

            try:
                plugin.run()
            except PluginRunError as ex:
                report = {
                    'plugin': {
                        'name': plugin.name,
                    }
                }
                if isinstance(ex, PluginRunTimedOut):
                    report['error'] = 'timed out'
                    report['timeout'] = ex.timeout
                elif isinstance(ex, PluginReportTooBig):
                    report['error'] = 'report too big'
                    report['limit'] = ex.limit
                    report['size'] = ex.size
                else:
                    report['error'] = 'died'
                    report['stdout'] = ex.stdout
                    report['stderr'] = ex.stderr
                    report['returncode'] = ex.returncode

                self._report(
                    self.compress({
                        'start': now,
                        'end': time.time(),
                        'incremental': False,
                        'name': 'HeartBeatPluginError',
                        'valid': self.cfg.plugin_error_reports.max_delivery_deadline,
                        'report': report,
                    }),
                    sendDelay=self.cfg.plugin_error_reports.max_delivery_delay
                )
            except gevent.GreenletExit:
                raise
            except:
                log.error('Unhandled plugin run() exception: %s', traceback.format_exc())
            else:
                if plugin.result:
                    plugin.result.setdefault('valid', plugin.config.max_delivery_deadline)
                    self._report(plugin.result, sendDelay=plugin.config.max_delivery_delay)

        finally:
            plugin.prev_run_timestamp = time.time()
    # }}}

    def _worker_loop(self):  # {{{
        log = self.log.getChild('scheduler')
        log.info('Started')
        runners = {}

        class LinkedExited(Exception):
            pass

        greenlet_exited_notifier = lambda grn, current=gevent.getcurrent(): current.throw(LinkedExited())

        while 1:
            try:
                # Search for new plugins and schedule them
                self._find_plugins()
                self._schedule_plugins()

                if not self._pluginRunQueue:
                    self.log.debug('No plugins scheduled to run at the moment, sleeping 30 seconds...')
                    gevent.sleep(30)
                    continue

                sleep_for = max(0, self._pluginRunQueue[0].next_run_timestamp - time.time())
                if sleep_for:
                    log.debug('Sleeping for %s.', util.td2str(sleep_for))
                    gevent.sleep(min(sleep_for, 10))
                    if sleep_for > 10:
                        continue

                plugin = self._pluginRunQueue.pop(0)
                if plugin in runners:
                    self.log.warn('Plugin %r still executing. Do not run it second time.', plugin.name)
                else:
                    greenlet = runners[plugin] = gevent.spawn(self._run_plugin, plugin)
                    greenlet.link(greenlet_exited_notifier)
                    log.debug('Scheduled plugin %r run.', plugin.name)

            except gevent.GreenletExit:
                log.info('Received stop signal')
                break
            except LinkedExited:
                for plugin, gt in runners.items():
                    if gt.ready():
                        if gt.successful():
                            log.debug('Plugin %r execution finished successfully.', plugin.name)
                        else:
                            log.error('Plugin %r execution failed.', plugin.name)
                        # schedule next turn here
                        if not plugin.config.repeat:
                            plugin.scheduled = True
                        else:
                            self._schedule_plugin(plugin)
                        runners.pop(plugin)
            except Exception:
                log.error('Unhandled exception: %s', formatException())
                os._exit(1)
    # }}}

    # Schedule, find and load plugins {{{
    def _schedule_plugins(self):
        for plugin_info in self._plugins.values():
            plugin = plugin_info[0]
            if plugin and not plugin.scheduled:
                self._schedule_plugin(plugin)

    def _schedule_plugin(self, plugin):
        plugin.compute_next_run()
        idx = bisect.bisect(
            [plugin_.next_run_timestamp for plugin_ in self._pluginRunQueue],
            plugin.next_run_timestamp
        )
        self._pluginRunQueue.insert(idx, plugin)
        plugin.log.debug('Scheduled #%d in queue' % (idx + 1, ))

    def _find_plugins_from_global_config(self):
        changed = False
        checked = set()

        if time.time() - self._global_config_check_time < self.CHECK_GLOBAL_CONFIG_INTERVAL:
            return changed

        self.log.debug('Looking for new/changed global plugins...')

        def _flatten(dct, name=None, defaults=None):
            result = {}
            defaults = defaults.copy() if defaults else {}
            is_group = '_defaults' in dct
            if is_group:
                defaults.update(dct['_defaults'])
                for subname, subdct in dct.items():
                    if subname == '_defaults':
                        continue
                    result.update(_flatten(subdct, subname, defaults))
            else:
                cfg = {}
                cfg.update(defaults)
                cfg.update(dct)
                result.update({name: cfg})

            return result

        cfg = libconfig.query('skynet.services.heartbeat-client', 'plugins', as_dict=True, evaluate=False)
        cfg = _flatten(cfg)

        for name, config in cfg.items():
            checked.add(name)
            changed = self._load_plugin_from_config(name, config) is not False or changed

        for name, (plugin, conf) in self._plugins.items():
            if not conf:
                # file plugin
                continue
            if name in checked:
                # already checked
                continue

            # Force to unload this plugin
            config['enabled'] = False
            changed = self._load_plugin_from_config(name, config) is not False or changed

        self._global_config_check_time = time.time()

        return changed

    def _find_plugins_from_files(self):
        checked = set()
        changed = False

        if time.time() - self._files_check_time < self.CHECK_FILES_INTERVAL:
            return changed

        self.log.debug('Looking for new/changed file plugins...')

        search_paths = (
            self.cfg.default_plugins.configs,
            self.cfg.user_plugins.configs
        )

        for possible_path in search_paths:
            possible_path = py.path.local(possible_path)
            if not possible_path.check(exists=1) or not possible_path.check(dir=1):
                continue

            for config_file in possible_path.listdir():
                checked.add(config_file)
                changed = self._load_plugin_from_file(config_file) is not False or changed

        for config_file in self._plugins_from_file.keys():
            if config_file in checked:
                continue

            if not config_file.check(exists=1):
                # This will force plugin to unload
                changed = self._load_plugin_from_file(config_file) is not False or changed

        self._files_check_time = time.time()

        return changed

    def _find_plugins(self):
        changed = False

        changed = self._find_plugins_from_global_config() or changed
        changed = self._find_plugins_from_files() or changed

        return changed

    def _load_plugin_from_file(self, config_file):
        name, conf_mtime = self._plugins_from_file.get(config_file, (None, 0))
        plugin, old_config = self._plugins.get(name, [None, None])

        try:
            if conf_mtime != config_file.stat().mtime:
                if not plugin:
                    self.log.info('Loading new plugin from %s', config_file)
                    plugin = Plugin.from_file(self.ctx, self.cfg, config_file)
                    plugin.prev_run_timestamp = self._state.get(plugin.name, None)
                else:
                    self.log.info('Reloading plugin from %s', config_file)
                    # Deschedule plugin first.
                    if plugin in self._pluginRunQueue:
                        self._pluginRunQueue.remove(plugin)
                    plugin.load(Plugin.load_config_from_file(self.cfg, config_file))

                if plugin.name in self._plugins and self._plugins[plugin.name][1] is not None:
                    self.log.info('Plugin "%s" already loaded from global config, skipping', plugin.name)
                    self._plugins_from_file[config_file] = (None, config_file.stat().mtime)
                else:
                    self._plugins[plugin.name] = (plugin, None)
                    self._plugins_from_file[config_file] = (plugin, config_file.stat().mtime)

                return True  # plugin list changed

            return False

        except Exception as ex:
            if plugin in self._pluginRunQueue:
                self._pluginRunQueue.remove(plugin)

            if plugin:
                self.log.debug('Removing plugin %s (%s)', config_file, ex)
                self._plugins.pop(plugin.name)
                if config_file in self._plugins_from_file:
                    self._plugins_from_file[config_file] = (None, self._plugins_from_file[config_file][1])
            else:
                import traceback
                self.log.warning('Not loading bad plugin from %s: %s', config_file, traceback.format_exc())
                try:
                    self._plugins_from_file[config_file] = [None, config_file.stat().mtime]
                except:
                    self._plugins_from_file.pop(config_file, None)
                    pass

            return True  # plugin list changed

    def _load_plugin_from_config(self, name, config):
        plugin, old_config = self._plugins.get(name, (None, None))
        try:
            if config != old_config:
                if not plugin:
                    if not config.get('enabled', True):
                        self.log.info('Not loading disabled plugin "%s" from global config', name)
                        return False

                    self.log.info('Loading new plugin "%s" from global config', name)
                    plugin = Plugin(name, self.ctx, self.cfg, config)
                    plugin.prev_run_timestamp = self._state.get(plugin.name, None)
                else:
                    if not config.get('enabled', True):
                        self.log.info('Unloading disabled plugin "%s" from global config', name)
                        if plugin in self._pluginRunQueue:
                            self._pluginRunQueue.remove(plugin)
                            self._plugins.pop(name)
                        return True

                    self.log.info('Reloading plugin "%s" from global config', name)
                    if plugin in self._pluginRunQueue:
                        self._pluginRunQueue.remove(plugin)
                    plugin.load(Plugin.load_raw_config(self.cfg, config, name=name))

                self._plugins[plugin.name] = (plugin, config)
                for config_file, (file_plugin, file_mtime) in self._plugins_from_file.items():
                    if file_plugin.name == name:
                        self._plugins_from_file[config_file] = (None, file_mtime)

                return True  # plugin list changed
            return False
        except Exception as ex:
            if plugin in self._pluginRunQueue:
                self._pluginRunQueue.remove(plugin)

            if plugin:
                self.log.debug('Removing plugin %s (%s)', plugin.name, ex)
                self._plugins.pop(plugin.name)
            else:
                import traceback
                self.log.warning('Not loading bad plugin %s from global config: %s', name, traceback.format_exc())

            return True  # plugin list changed
    # Schedule, find and load plugins }}}
# }}}


class Plugin(object):  # {{{
    defaults = None

    # Init and load {{{
    def __init__(self, name, ctx, cfg, conf):
        self.name = name
        self.ctx = ctx
        self.cfg = cfg

        Plugin._set_defaults(cfg)
        conf = Plugin.load_raw_config(cfg, conf, name=name)

        self.load(conf)

    @classmethod
    def _set_defaults(cls, cfg):
        if cls.defaults is None:
            cls.defaults = {}
            for key, value in cfg.iteritems():
                fkey = '%s.default_options.' % (cfg[key], )
                if key.startswith(fkey):
                    cls.defaults[key[len(fkey):]] = value

    @classmethod
    def load_raw_config(cls, cfg, config, name=None):
        cls._set_defaults(cfg)

        new_config = AppConfig()
        new_config.load(config)

        config = new_config

        for key, value in Plugin.defaults.items():
            if key not in config:
                config[key] = value

        if name is None:
            name = config.name

        if 'logname' not in config:
            config['logname'] = name.lower()

        config['logname'] = config.logname
        return config

    @classmethod
    def load_config_from_file(cls, cfg, config_file):
        cls._set_defaults(cfg)

        config = yaml.load(config_file.open('rb'))

        return cls.load_raw_config(cfg, config)

    @classmethod
    def from_file(cls, ctx, cfg, config_file):
        config = cls.load_config_from_file(cfg, config_file)
        return cls(config['name'], ctx, cfg, config)

    def load(self, config):
        self.config = config
        self.path = None
        self.result = None
        self.next_run_timestamp = None
        self.prev_run_timestamp = None
        self.prevRunFinishTimestamp = None
        self.prevRunResult = None
        self.prevRunResultTimestamp = None
        self.scheduled = False
        self.madeRetries = 0
        self._compressor = None

        self.log = self.ctx.log.getChild('plugin').getChild(self.config.logname)

        if self.config.returns not in ('raw', 'stdout', 'stderr', 'msgpack', ):
            raise Exception('Unknown return type: %s' % (self.config.returns, ))

        path = None

        if '/' not in self.config.executable:
            for path in (
                py.path.local(self.cfg.default_plugins.binaries).join(self.config.executable),
                py.path.local(self.cfg.user_plugins.binaries).join(self.config.executable),
            ):
                if not path.check(exists=1):
                    path = None
                else:
                    break

            # If we didnt find it yet, maybe it is in PATH?
            if path is None:
                path = py.path.local.sysfind(self.config.executable)

            if path is None:
                raise Exception('Cant find executable by relative path')
        else:
            path = py.path.local(self.config.executable)
            if not path.check(exists=1):
                raise Exception('Executable file %s does not exists', path)

        self.path = path
        args = getattr(self.config, 'args', None)
        if isinstance(args, basestring):
            self.config.args = args.split()
        elif args is None:
            self.config.args = []

        assert type(self.config.timeout) in (int, float), \
            'Invalid <timeout> key: %r' % (self.config.timeout, )
        assert self.config.timeout <= 600, \
            'Timeouts > 10 minutes are not supported, got %s' % util.td2str(self.config.timeout)

        if self.config.compression:
            meth, level = self.config.compression.split(':')
            self.compression = meth
            if meth == 'zip':
                def compressor(data):
                    return zlib.compress(data, int(level))
                self._compressor = compressor
            elif meth == 'bz2':
                def compressor(data):
                    return bz2.compress(data, int(level))
                self._compressor = compressor
            else:
                raise Exception('Unknown compression %r configured.' % self.config.compression)

        return True
    # Init and load }}}

    def compress(self, report):
        if not self._compressor:
            return report
        report['report'] = self._compressor(report['report'])
        report['compression'] = self.compression
        return report

    def compute_next_run(self):  # {{{
        now = time.time()
        prev_run_timestamp = self.prev_run_timestamp if self.prev_run_timestamp is not None else now
        if self.scheduled or self.config.first_run_delay is True:
            run_every = self.config.run_every
            runMin, runMax = (run_every, run_every) if isinstance(run_every, (int, float)) else run_every
        else:
            if self.config.first_run_delay is False:
                self.config.first_run_delay = 0
            run_delay = self.config.first_run_delay
            runMin, runMax = (run_delay, run_delay) if isinstance(run_delay, (int, float)) else run_delay

        self.next_run_timestamp = max(
            now,
            prev_run_timestamp + (0 if runMin == runMax == 0 else (random.random() * (runMax - runMin) + runMin))
        )
        if self.next_run_timestamp <= now:
            self.log.info('Computed next run as soon as possible.')
        else:
            self.log.info('Computed next run in %s from now.', util.td2str(self.next_run_timestamp - now))

        self.scheduled = True
    # }}}

    def run(self):  # {{{
        try:
            cmd = [self.path.strpath]
            cmd += self.config.args
            # here is big nice awful crutch since plugins' config is totally independent from
            # global config and we cannot just evaluate variables
            cmd = [part.replace("${WorkdirPath}", self.ctx.cfg.WorkdirPath) for part in cmd]
            self.log.debug('Running %s' % (cmd, ))

            now = time.time()
            proc = subproc.Popen(cmd, stdout=subproc.PIPE, stderr=subproc.PIPE, close_fds=True, log=self.log)
            with gevent.Timeout(self.config.timeout):
                try:
                    stdout, stderr = proc.communicate()
                except gevent.Timeout as ex:
                    pass

            self.log.debug('Done run in %0.4g seconds' % (time.time() - now, ))

            if proc.returncode == 0:
                if self.config.returns in ['msgpack', 'stdout']:
                    data = stdout
                elif self.config.returns == 'stderr':
                    data = stderr
                elif self.config.returns == 'raw':
                    data = msgpack.dumps({
                        'stdout': stdout,
                        'stderr': stderr,
                        'returncode': self.config.returns,
                    })

                if len(data) > self.config.max_report_size:
                    ex = PluginReportTooBig(len(data), self.config.max_report_size)
                    self.log.error(str(ex))
                    raise ex

                self.result = self.compress({
                    'incremental': self.config.incremental,
                    'format': self.config.returns,
                    'name': self.name,
                    'start': now,
                    'end': time.time(),
                    'report': data,
                })

                if self.config.force_report_every:
                    if isinstance(self.config.force_report_every, (list, tuple)):
                        forceReportEveryMin, forceReportEveryMax = self.config.force_report_every
                        self.result['force'] = (
                            random.random() * (forceReportEveryMax - forceReportEveryMin) +
                            forceReportEveryMin
                        )
                    else:
                        self.result['force'] = self.config.force_report_every
                if not self.config.report_only_if_changes:
                    self.result['force'] = float('-inf')
            else:
                if proc.returncode is None:
                    self.log.error('Process %s exited by timeout %r seconds', self.path, self.config.timeout)
                    ex = PluginRunTimedOut('Timed out %r seconds' % (self.config.timeout, ))
                    ex.timeout = self.config.timeout
                else:
                    self.log.error('Process %s exited with status: %d', self.path, proc.returncode)
                    self.log.error('STDERR: %s', stderr)
                    ex = PluginRunError('Return code: %d' % (proc.returncode, ))
                    ex.returncode = proc.returncode
                    ex.stdout = stdout
                    ex.stderr = stderr
                raise ex
        except gevent.GreenletExit:
            raise
        except Exception as ex:
            if hasattr(ex, 'retcode'):
                self.prevRunResult = [ex.retcode, ex.stderr]
            else:
                self.prevRunResult = [ex, str(ex)]

            if isinstance(ex, PluginRunError):
                raise
            else:
                self.log.warning('Got error while calling plugin: %s', formatException())

        finally:
            try:
                proc.kill()
                proc.wait()
            except:
                pass

            self.schedulerState = time.time()
    # }}}

    # Misc meths {{{
    def __repr__(self):
        return '<Plugin: %s>' % (self.name, )
    # Misc meths }}}
# }}}
