#!/skynet/python/bin/python
"""
Plugin which collects host system information:
    * OS name
    * OS version
    * OS arch
    * CPU model (e.g. E5640)
    * RAM size (in kb)
    * total space in /
    * total space in /db/BASE
    * total space in /var/tmp
"""

from __future__ import absolute_import, print_function, division

import os
import sys
import re
import argparse
import subprocess
import pkg_resources
import platform

pkg_resources.require('msgpack-python')
import msgpack

if sys.platform != 'cygwin':
    pkg_resources.require('psutil')
    import psutil


def run_command(*args):
    p = subprocess.Popen(*args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
    stdout, stderr = p.communicate()
    if p.returncode:
        raise Exception()
    return stdout


def _lsb_release():
    try:
        os_name, os_version = run_command(['lsb_release -irs']).strip().split('\n')
    except Exception:
        return None
    else:
        return os_name, os_version


def _fix_field_val(val):
    if val.startswith('"'):
        return val[1:-1]
    return val


def _common_heuristic():
    os_release_files = {
        '/etc/os_release': ('NAME', 'VERSION'),
        '/etc/os-release': ('NAME', 'VERSION'),
        '/etc/lsb-release': ('DISTRIB_ID', 'DISTRIB_RELEASE'),
    }
    for fn, fields in os_release_files.iteritems():
        if os.path.exists(fn):
            try:
                info = dict(field.split('=', 1) for field in open(fn).read().strip().split('\n'))
            except (EnvironmentError, ValueError):
                continue
            if fields[0] in info and fields[1] in info:
                return _fix_field_val(info[fields[0]]), _fix_field_val(info[fields[1]])


def _freebsd_version():
    if sys.platform.lower().startswith('freebsd'):
        name, _, version, _, _ = os.uname()
        return name, version


def _macos_version():
    if sys.platform.lower() == 'darwin':
        name = 'Mac OS X'
        version = platform.mac_ver()[0]
        return name, version


def _win_version():
    if sys.platform.lower() == 'cygwin':
        # FIXME (torkve) Windows and its encodings are just a big piece of holyshit
        # so if you can debug the output, debug it yourself!
        # data = run_command(['wmic os get Caption,CSDVersion /value'])
        return "Cygwin", "Unknown"


def _unix_heuristic():
    unix_release_files = [
        ('/etc/debian_version', 'Debian'),
        ('/etc/SuSE-release', 'SuSE'),
        ('/etc/redhat-release', 'RedHat Linux'),
    ]
    for filename, name in unix_release_files:
        if os.path.exists(filename):
            try:
                version = open(filename).read().strip()
                return name, version
            except EnvironmentError:
                continue


def os_info():
    kernelName, _, kernelVersion, _, kernelArch = os.uname()
    os_info = (
        _lsb_release()
        or _freebsd_version()
        or _macos_version()
        or _win_version()
        or _common_heuristic()
        or _unix_heuristic()
        or ('Unknown', 'Unknown')
    )

    res = {
        'name': kernelName,
        'version': kernelVersion,
        'arch': kernelArch,
        'os_name': os_info[0],
        'os_version': os_info[1],
    }

    return res


def disk_info():
    interested = ['/', '/db/BASE', '/var/tmp']

    def dir_info(catalog):
        try:
            return None if sys.platform == 'cygwin' else psutil.disk_usage(catalog).total
        except EnvironmentError:
            return None

    res = {}
    for catalog in interested:
        res[catalog] = dir_info(catalog)
    return res


def cpu_info():
    def grep_model(s):
        # common amd processor
        if s.find('AMD') != -1:
            return 'AMD' + s.rstrip().rpartition(' ')[2]
        # common intel processor
        if s.find('CPU') != -1:
            return s[s.find('CPU') + 3:].strip().split(' ')[0]
        else:
            return None

    osname = os.uname()[0]
    if osname == 'Linux':
        data = run_command(["cat /proc/cpuinfo"])
        m = re.search('model name([^\n]*)', data)
        if not m:
            return {'error': "no model name in /proc/cpuinfo"}
        data = m.group(0)

        cpu_freq = run_command(["cat /proc/cpuinfo | grep 'model name' | awk '{print$NF}'" +
                                "| awk -F 'GHz' '{print($1 * 1000)}' | sort -n | head -1"])
        cpu_freq = int(cpu_freq.strip())

        max_cpu_freq = run_command(["cat /proc/cpuinfo | grep 'cpu MHz' | awk '{print$NF}' | sort -rn | head -1"])
        max_cpu_freq = int(float(max_cpu_freq.strip()))

    elif osname == 'FreeBSD':
        try:
            data = run_command(["sysctl hw.model"])
        except ValueError:
            return {'error': 'Unsupported FreeBSD version %r.' % os.uname()[2]}

        cpu_freq = run_command(["sysctl dev.cpu.0.freq | awk '{print$NF}'"])
        cpu_freq = int(cpu_freq.strip())
        max_cpu_freq = cpu_freq

    else:
        return {'error': "unsupported os '{0}'".format(osname)}

    return {'model': grep_model(data), 'cpu_freq': cpu_freq, 'max_cpu_freq': max_cpu_freq}


def mem_info():
    return {'total_ram': 0 if sys.platform == 'cygwin' else psutil.phymem_usage().total}


def host_info():
    return {
        'os': os_info(),
        'hwinfo': {
            'mem': mem_info(),
            'cpu': cpu_info(),
            'disk': disk_info()
        }
    }


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    return parser.parse_args()


def main():
    args = parse_args()
    result = host_info()
    if args.format == 'pretty':
        print(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
