#!/skynet/python/bin/python

import re
import urllib2
import os
import json
import socket
import subprocess
import time
import random
import argparse
from pprint import pprint
import msgpack
import fcntl

class HostFlags:
    IS_VM_GUEST=0

def run_command(args, raise_failed=True, timeout=None, sleep_timeout=1., cwd=None, close_fds=False, stdin=None):
    try:
        if stdin is None:
            p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, cwd=cwd, close_fds=close_fds)
        else:
            p = subprocess.Popen(args, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE, cwd=cwd, close_fds=close_fds)
    except Exception, e:
        if raise_failed:
            raise Exception, "subpocess.Popen for <<%s>> failed: %s" % (args, e)
        else:
            return (1, 'Got unknown error %s' % e, '')

    try:
        if not stdin is None:
            p.stdin.write(stdin) # Is it correct ?
            p.stdin.close()

        if timeout is None:
            out, err = p.communicate()
            if p.returncode != 0 and raise_failed:
                raise Exception, "Command <<%s>> returned %d\nStdout:%s\nStderr:%s" % (args, p.returncode, out, err)
        else:
            out, err = '', ''
            wait_till = time.time() + timeout

            fcntl.fcntl(p.stdout, fcntl.F_SETFL, fcntl.fcntl(p.stdout, fcntl.F_GETFL) | os.O_NONBLOCK)
            fcntl.fcntl(p.stderr, fcntl.F_SETFL, fcntl.fcntl(p.stderr, fcntl.F_GETFL) | os.O_NONBLOCK)

            while time.time() < wait_till:
                p.poll()
                try:
                    while True:
                        r = os.read(p.stdout.fileno(), 1024)
                        out += r
                        if len(r) == 0:
                            break
                except OSError:
                    pass
                try:
                    while True:
                        r = os.read(p.stderr.fileno(), 1024)
                        err += r
                        if len(r) == 0:
                            break
                except OSError:
                    pass
                if p.returncode == 0:
                    return (p.returncode, out, err)
                if p.returncode != None:
                    if raise_failed:
                        raise Exception, "Command <<%s>> returned %d\nStdout:%s\nStderr:%s" % (args, p.returncode, out, err)
                    else:
                        return (p.returncode, out, err)
                time.sleep(sleep_timeout)

            if raise_failed:
                raise Exception, "Command <<%s>> timed out (%f seconds)" % (args, timeout)
    finally:
        if p.returncode is None:
            p.kill()
            p.wait()

    return p.returncode, out, err

def get_url_json_content(url):
    try:
        result = urllib2.urlopen(url).read().strip()
        if result == "":
            raise Exception, "Got empty result while loading %s" % url
        return json.loads(result)
    except urllib2.HTTPError:
        raise Exception, "Got HTTPError while loading %s" % url

def _get_primary_disk():
    if os.uname()[0] == "Linux":
        data = run_command(["df", "-P", "-B", "1073741824"])[1]
    if os.uname()[0] == "FreeBSD":
        data = run_command(["df", "-g"])[1]

    data = map(lambda x: (re.split(' +', x)[0], int(re.split(' +', x)[1])), data.split('\n')[1:-1])
    data = filter(lambda (name, sz): name.startswith('/dev'), data)
    data.sort(cmp = lambda (name1, sz1), (name2, sz2): cmp(sz2, sz1))
    return data[0][0]

def hivemind_hostname(parent):
    if 'custom_hostname' in parent.params:
        return parent.params['custom_hostname']

    try:
        # for some old machines with IPv4 this magic somehow works better
        return socket.gethostbyaddr(socket.gethostbyname(socket.gethostname()))[0]
    except:
        return socket.gethostname()

def is_vm_guest():
    os_type = os.uname()[0]
    if os_type not in ['Linux', 'FreeBSD']:
        raise Exception, 'Unknown os type "%s"' % os_type
    if os_type == 'FreeBSD':
        # Not implemented actually
        return False
    scsi = open('/proc/scsi/scsi').read()
    return scsi.find('QEMU HARDDISK') != -1

class LinuxDiskInfo:
    MARKERS = [ ('Host:', 'host'),
                ('Vendor:', 'vendor'),
                ('Model:', 'model'),
                ('Type:', 'atype'),
                ('ANSI  SCSI revision:', 'revision'),
              ]
    def __init__(self, data):
        indexes = map(lambda (marker, field): data.find(marker), self.MARKERS)
        if -1 in indexes:
            raise Exception, "No some data in disk description"
        indexes.append(len(data))

        for i in range(len(self.MARKERS)):
            self.__dict__[self.MARKERS[i][1]] = data[indexes[i] + len(self.MARKERS[i][0]):indexes[i+1]].strip()

class GetStats(object):
    def __init__(self, models, funcs, user = None, timeout = 1, ignore_detect_fail = False):
        self.params = { 'models' : models, }
        self.funcs = funcs
        self.osUser = user
        self.timeout = timeout
        self.ignore_detect_fail = ignore_detect_fail
        self.warnings = []

    def anti_ddos_delay(self):
        delay = random.randrange(self.timeout)
        time.sleep(delay)

    def run(self):
        self.anti_ddos_delay()
        result = {}
        for signal, func in self.funcs.iteritems():
            try:
                result[signal] = func(self)
            except:
                if self.ignore_detect_fail:
                    continue
                else:
                    raise

        return result, self.warnings

    @classmethod
    def member_count(cls):
        return 8


def detect_disk(parent):
    os_type = os.uname()[0]

    if os_type == 'Linux':
        sz = 0
        for device in os.listdir('/sys/block'):
            if device.startswith('loop') or device.startswith('ram'):
                continue
            is_rotational = int(open(os.path.join('/sys/block', device, 'queue/rotational')).read())
            real_device_path = os.readlink(os.path.join('/sys/block', device))
            if is_rotational > 0 and not real_device_path.startswith('../devices/virtual/'):
                sz += int(open(os.path.join('/sys/block', device, 'size')).read()) * 512 / 1024 / 1024 / 1024
        return sz

    elif os_type == 'FreeBSD':
        return 0

    else:
        raise Exception, "Unknown os %s" % os.uname()[0]

def detect_flags(parent):
    return (1 << HostFlags.IS_VM_GUEST) * bool(is_vm_guest())

def detect_ipmi(parent):
    ipminame = '%s.ipmi.yandex-team.ru' % hivemind_hostname(parent)
    try:
        result = socket.getaddrinfo(ipminame, None, socket.AF_INET6)
        if len(result) == 0:
            return 0
        else:
            return 1
    except socket.gaierror:
        return 0

def detect_ipv4addr(parent):
    myname = hivemind_hostname(parent)
    m = re.search('has address (.*)', run_command(['host', myname])[1])
    if m is not None:
        return m.group(1)
    else:
        return 'unknown'

def detect_ipv6addr(parent):
    myname = hivemind_hostname(parent)
    m = re.search('has IPv6 address (.*)', run_command(['host', myname])[1])
    if m is not None:
        return m.group(1)
    else:
        PDOMAIN = '.yandex.ru'
        if myname.endswith(PDOMAIN):
            myname = myname[:-len(PDOMAIN)] + '.search.yandex.net'
            m = re.search('has IPv6 address (.*)', run_command(['host', myname])[1])
            if m is not None:
                return m.group(1)

    return 'unknown'

def detect_issue(parent):
    if os.uname()[0] == "Linux":
        return open("/etc/issue").read().strip()
    elif os.uname()[0] == "FreeBSD":
        return run_command(["uname", "-rs"])[1].strip()
    else:
        return "unknown"

def detect_kernel(parent):
    return run_command(["uname", "-r"])[1].strip()

def detect_memory(parent):
    if os.uname()[0] == "Linux":
        mem = int(re.search('MemTotal: ([^\n]*)',run_command(["cat", "/proc/meminfo"])[1]).group(1).strip().split(' ')[0]) / 1024 / 1024
    if os.uname()[0] == "FreeBSD":
        mem = int(run_command(["sysctl", "vm.stats.vm.v_page_count"])[1].strip().split(' ')[1]) * 4096 / 1024 / 1024 / 1024

    if mem % 8 <4:
        mem = mem - mem % 8
    else:
        mem += 8 - mem % 8

    return mem

def detect_model(parent):
    BOT_MODELS = {
        "OPTERON6172" : "AMD6172",
        "OPTERON6176" : "AMD6176",
        "OPTERON6274" : "AMD6274",
        "XEONE5-2650V2" : "E5-2650v2",
        "XEONE5-2660" : "E5-2660",
        "XEONE5-2667V" : "E5-2667",
        "XEONE5-2667V2" : "E5-2667v2",
        "XEON5440" : "E5440",
        "XEON5530" : "E5530",
        "XEON5620" : "E5620",
        "XEON5645" : "E5645",
        "XEON5410" : "L5410",
        "XEON5450" : "X5450",
        "XEON5670" : "X5670",
        "XEON5675" : "X5675",
    }

    if "custom_hostname" in parent.params: # local run for arbitary host:
        response = retry_urlopen(3, "http://bot.yandex-team.ru/api/consistof.php?name=%s" % parent.params["custom_hostname"]).read()
        m = re.search("Model:\s*(.*?)\s+", response)
        bot_model = m.group(1)
        return BOT_MODELS[bot_model]
    else:
        if os.uname()[0] == "Linux":
            model = re.search('model name\s+: (.*)', run_command(["cat", "/proc/cpuinfo"])[1]).group(1).strip()
        if os.uname()[0] == "FreeBSD":
            model = run_command(["sysctl", "-n", "hw.model"])[1].strip()

        if model in parent.params['models']:
            return parent.params['models'][model]
        else:
            raise Exception, "Unknown cpu model <%s>" % model

def detect_n_disks(parent):
    if os.uname()[0] == "Linux":
        __, n_disks, _ = run_command(["cat", "/proc/scsi/scsi"])
        n_disks = n_disks.split('\n')
        n_disks = [line.strip() for line in n_disks if line.strip()]
        assert(n_disks[0].startswith('Attached devices:'))
        n_disks = n_disks[1:]
        assert(len(n_disks) % 3 == 0)

        disks = map(lambda (x, y, z): LinuxDiskInfo(' '.join([x, y, z])), zip(n_disks[0::3], n_disks[1::3], n_disks[2::3]))

        # filter stuff
        disks = filter(lambda x: not x.model.startswith('Virtual'), disks)
        disks = filter(lambda x: x.atype != 'CD-ROM', disks)
        disks = filter(lambda x: x.model.find('SSD') == -1, disks)

        return len(disks)

    assert(False)

def detect_name(parent):
    return hivemind_hostname(parent)

def detect_os(parent):
    return os.uname()[0]

def detect_raid(parent):
    diskname = _get_primary_disk()
    shortname = diskname.split('/')[-1]
    if os.uname()[0] == "Linux":
        data = open('/proc/mdstat').read()
        m = re.search('%s : (.+?) (.+?) ' % shortname, data)
        if m:
            return m.group(2)
        else:
            return "raid0"
    else:
        return "unknown"

def detect_ssd(parent):
    os_type = os.uname()[0]
    if os_type not in ['Linux', 'FreeBSD']:
        raise Exception, 'Unknown os type "%s"' % os_type
    is_bsd = os_type == 'FreeBSD'
    if is_bsd:
        _1, output, _2 = run_command(['df', '-g'])
    else:
        _1, output, _2 = run_command(['df', '-P', '-B', '1073741824'])
    lines = output.rstrip().split('\n')[1:]
    lines = [line.split() for line in lines]
    mountpnt_size = dict((line[5], ((int(float(line[1]))), line[0].split('/')[-1])) for line in lines)

    # we have a convention, that if there is ssd disk
    # it will be mounted at '/ssd' or '/fasthd'
    if '/ssd' in mountpnt_size:
        return str(mountpnt_size['/ssd'][0])
    if '/fasthd' in mountpnt_size:
        return str(mountpnt_size['/fasthd'][0])


    # did not found good one, go though all devices
    if os_type == 'Linux':
        sz = 0
        for device in os.listdir('/sys/block'):
            if device.startswith('loop') or device.startswith('ram'):
                continue
            is_rotational = int(open(os.path.join('/sys/block', device, 'queue/rotational')).read())
            if not is_rotational:
                sz += int(open(os.path.join('/sys/block', device, 'size')).read()) * 512 / 1024 / 1024 / 1024
        return sz


def detect_all():
    stats_getter = GetStats({ 'X5450' : 'X5450', 'Intel(R) Xeon(R) CPU E3-1265L v3 @ 2.50GHz' : 'E3-1265Lv3', 'Intel(R) Xeon(R) CPU           E5630  @ 2.53GHz' : 'E5630', 'E5410' : 'E5410', 'Intel(R) Xeon(R) CPU E5-2640 v3 @ 2.60GHz' : 'E5-2640v3', 'AMD Opteron(tm) Processor 6168' : 'AMD6168', 'E5345' : 'E5345', 'L5410' : 'L5410', 'E5540' : 'E5540', 'AMD Opteron(tm) Processor 6238' : 'AMD6238', 'Intel(R) Xeon(R) CPU           E5620  @ 2.40GHz' : 'E5620', 'E3-1280v3' : 'E3-1280v3', 'unknown' : 'unknown', 'Intel(R) Xeon(R) CPU           X5675  @ 3.07GHz' : 'X5675', 'X5365' : 'X5365', 'Intel(R) Xeon(R) CPU E5-2697 v3 @ 2.60GHz' : 'E5-2697v3', 'Intel(R) Xeon(R) CPU E5-2650L v3 @ 1.80GHz' : 'E5-2650Lv3', 'C2750' : 'C2750', 'Intel(R) Xeon(R) CPU E3-1230L v3 @ 1.80GHz' : 'E3-1230Lv3', 'Intel(R) Xeon(R) CPU           E5645  @ 2.40GHz' : 'E5645', 'AMD Opteron(tm) Processor 6176' : 'AMD6176', 'Intel(R) Xeon(R) CPU E5-2660 v2 @ 2.20GHz' : 'E5-2660v2', 'Intel(R) Xeon(R) CPU E5-2660 v3 @ 2.60GHz' : 'E5-2660v3', 'Intel(R) Xeon(R) CPU           E5530  @ 2.40GHz' : 'E5530', 'Intel(R) Xeon(R) CPU E5-2667 0 @ 2.90GHz' : 'E5-2667', 'Intel(R) Xeon(R) CPU           E5440  @ 2.83GHz' : 'E5440', 'AMD Opteron(tm) Processor 6172' : 'AMD6172', 'Intel(R) Xeon(R) CPU E5-2660 0 @ 2.20GHz' : 'E5-2660', 'E5462' : 'E5462', 'E3-1275v3' : 'E3-1275v3', 'Intel(R) Xeon(R) CPU E5-2650 v3 @ 2.30GHz' : 'E5-2650v3', 'Intel(R) Xeon(R) CPU E5-2650 v2 @ 2.60GHz' : 'E5-2650v2', 'Intel(R) Xeon(R) CPU E5-2643 v3 @ 3.40GHz' : 'E5-2643v3', 'Intel(R) Xeon(R) CPU E5-2667 v3 @ 3.20GHz' : 'E5-2667v3', 'Intel(R) Xeon(R) CPU           X5670  @ 2.93GHz' : 'X5670', 'Intel(R) Xeon(R) CPU E5-2652 v3 @ 2.30GHz' : 'E5-2652v3', 'AMD Opteron(TM) Processor 6274' : 'AMD6274', 'Intel(R) Xeon(R) CPU E5-2667 v2 @ 3.30GHz' : 'E5-2667v2' }, { 'disk' : detect_disk, 'flags' : detect_flags, 'ipmi' : detect_ipmi, 'ipv4addr' : detect_ipv4addr, 'ipv6addr' : detect_ipv6addr, 'issue' : detect_issue, 'kernel' : detect_kernel, 'memory' : detect_memory, 'model' : detect_model, 'n_disks' : detect_n_disks, 'name' : detect_name, 'os' : detect_os, 'raid' : detect_raid, 'ssd' : detect_ssd }, ignore_detect_fail = True)
    return stats_getter.run()

# ===============================================================================
# main stuff
# ===============================================================================
def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'binary'), default='pretty')
    return parser.parse_args()

def main():
    args = parse_args()

    hwdata, warnings = detect_all()

    if args.format == 'pretty':
        pprint(hwdata)
    elif args.format == 'binary':
        print(msgpack.packb(hwdata))

if __name__ == '__main__':
    main()
