from __future__ import absolute_import, print_function, division

import argparse
import glob
import json
import msgpack
import os
import py
import socket
import time
import re
import requests

from pprint import pprint
from xml.dom.minidom import parseString
import subprocess


class Keys():
    is_alive = 'a'  # active configuration, alive instance
    bsconfig_sync_on = 'bs'  # 0/1 - syncdisable,syncshardsdisable/enable bsconfig sync
    configurations = 'c'  # list of all configurations
    instances = 'i'  # list of all instances
    models = 'm'  # ranking models of base search instance
    md5 = 'md5'  # instance binary's md5
    multiplier = 'mu'  # base search instance multiplier (ruchka) - /supermind?action=getmult
    models_md5 = 'mv'  # ranking models MD5 of base search instance
    is_prepared = 'p'  # prepared configuration, prepared instance
    revision = 'r'  # instance binary's svn revision
    shard_name = 's'  # instance's shard name
    sandbox_task = 'st'  # instance's binary sandbox task id
    itype = 't'  # instance type
    all_tags = 'tags'  # all instance tags
    report_version = 'v'  # report version
    svn_url = 'u'  # instance binary's svn path
    other_resources = 'res'  # info on other instance resources in a dictionary
    conf = 'conf'  # configuration id


REPORT_VERSION = 3
CONFIG_INSTALL_PATH = py.path.local('/db/bsconfig/configinstall')
CONFIG_ACTIVE_PATH = py.path.local('/db/bsconfig/configactive')
DB_BSCONFIG_STATE = py.path.local('/db/bsconfig/state')
INSTANCE_FIELDS = [
    Keys.is_alive, Keys.models, Keys.md5, Keys.multiplier, Keys.models_md5, Keys.is_prepared, Keys.revision,
    Keys.shard_name, Keys.svn_url, Keys.configurations, Keys.sandbox_task, Keys.itype, Keys.other_resources,
    Keys.all_tags, Keys.conf
]
ISS_AGENT_CONFIGURATIONS_PATH = '/db/iss3/configurations/'
ISS_AGENT_INFO_PORT = 25536
ISS_AGENT_HTTP_TIMEOUT = 20

USER_AGENT = "Heartbeat/Skynet"

YANDSEARCH_AWARE_ITYPES = {'addrsmiddle', 'addrsupper', 'audioupper', 'ban', 'base', 'fusion', 'int', 'intl2',
                           'intsaas', 'miscbase', 'miscmmeta', 'mmeta', 'noapache', 'none', 'qsuserdatabase',
                           'querysearchbase', 'realsearch', 'rtyserver', 'unknown'}


# {'bs': 1,
# 'c': {'production_base-1364211362': {'a': 1,
# 'p': 1}},
# 'i': {'ws10-100:7300': {'a': 1,
# 'm': {'2full.by.latin:onenormmn5421u2243c70c005:': 'ByLatin_onenormmn5421u2243c70c005_',
#                               'full.ua:mn3415wopol:': 'Ua_mn3415_'},
#                         'md5': '4ac2aad79bcf6b9d7ec2231965b06cb2',
#                         'mu': 0.0,
#                         'mv': '09be315a77e8b001d9a39f22a59a4b29',
#                         'p': 1,
#                         'r': 987709,
#                         's': 'primus118-022-1364211362',
#                         'st': 31462428,
#                         't': 'base',
#                         'u': 'svn+ssh://arcadia.yandex.ru/arc/tags/base/stable-23-12/arcadia'}},
# 'v': 3}


class AgentException(StandardError):
    pass


def load_iss_instance_states(url):
    """
    load json data and transform unicode to string
      http://stackoverflow.com/questions/956867/
      http://stackoverflow.com/a/6633651/116373
    """

    def _decode_list(data):
        rv = []
        for item in data:
            if isinstance(item, unicode):
                item = item.encode()
            elif isinstance(item, list):
                item = _decode_list(item)
            elif isinstance(item, dict):
                item = _decode_dict(item)
            rv.append(item)
        return rv

    def _decode_dict(data):
        rv = {}
        for key, value in data.iteritems():
            if isinstance(key, unicode):
                key = key.encode()
            if isinstance(value, unicode):
                value = value.encode()
            elif isinstance(value, list):
                value = _decode_list(value)
            elif isinstance(value, dict):
                value = _decode_dict(value)
            rv[key] = value
        return rv

    json_data = get_info_by_http(socket.getfqdn(), ISS_AGENT_INFO_PORT, url=url, timeout=ISS_AGENT_HTTP_TIMEOUT)
    try:
        return json.loads(json_data, object_hook=_decode_dict)
    except (ValueError, IndexError):
        raise AgentException()


def get_iss_data():
    all_instances = load_iss_instance_states('/v1/instances')
    # print("=== all ===")
    # pprint(all_instances)

    active_instances = load_iss_instance_states('/v1/instances/active')
    # print("=== active ===")
    # pprint(active_instances)

    short_host = set(x['slot']['host'].split('.', 1)[0] for x in active_instances.values()).pop()

    # prepare instance path map
    files_depth3 = glob.glob(ISS_AGENT_CONFIGURATIONS_PATH + '/*/*/*')  # root / family / config / port(service)
    dirs_depth3 = filter(lambda f: os.path.isdir(f), files_depth3)

    instance_path_map = {}

    for instance_path in dirs_depth3:
        _family, configuration, port = instance_path.split(os.path.sep)[-3:]
        short_host_port = '%s:%s' % (short_host, port)
        instance_path_map[(short_host_port, configuration)] = os.path.realpath(instance_path)

    instances = {}

    # 1. add all instances
    for data in all_instances:
        port = data['slot']['service']
        short_host_port = '{}:{}'.format(short_host, port)
        configuration = '{}#{}'.format(data['configurationId']['family'], data['configurationId']['id'])
        instance_key = (short_host_port, configuration)
        mtn_ip = data['ipAddress'] if data['ipAddress'] != 'IP_IS_NOT_SET' else ''

        instance = {
            Keys.is_alive: 0,
            Keys.models: {},
            Keys.multiplier: 0.0,
            Keys.is_prepared: 0,
            Keys.shard_name: '',
            Keys.configurations: '',
            'host': short_host,
            'port': port,
            'name': short_host_port,
            'instance_dir': instance_path_map.get(instance_key, ''),
            'conf': configuration,
            'mtn_ip': mtn_ip,
        }

        instances[short_host_port] = instance

    # 2. add active instances, can overwrite some of all instances
    for host_port, data in active_instances.iteritems():
        port = data['slot']['service']
        short_host_port = '{}:{}'.format(short_host, port)
        configuration = '{}#{}'.format(data['configurationId']['family'], data['configurationId']['id'])
        instance_key = (short_host_port, configuration)
        mtn_ip = data['ipAddress'] if data['ipAddress'] != 'IP_IS_NOT_SET' else ''

        instance = {
            Keys.is_alive: 1,
            Keys.models: {},
            Keys.multiplier: 0.0,
            Keys.is_prepared: 1,
            Keys.shard_name: '',
            Keys.configurations: '',
            'host': short_host,
            'port': port,
            'name': short_host_port,
            'instance_dir': instance_path_map.get(instance_key, ''),
            'conf': configuration,
            'mtn_ip': mtn_ip,
        }

        # add itype
        properties = data.get('entityDescription', {}).get('properties', {})
        tags = properties.get('tags', '')
        tags = ensure_gencfg_tags(tags, properties)
        itype = [tag for tag in tags.split() if tag.startswith('a_itype_')]
        if itype:
            itype = itype[0].replace('a_itype_', '')
            instance[Keys.itype] = itype
        instance[Keys.all_tags] = tags

        instances[short_host_port] = instance

    configurations = {}

    # fill configurations statistics
    for instance in instances.itervalues():
        conf = instance['conf']
        conf_stat = configurations.setdefault(conf, {Keys.is_alive: 0, Keys.is_prepared: 0,
                                                     'a_count': 0, 'p_count': 0,
                                                     'i_count': 0, 'is_iss': 1})
        conf_stat['i_count'] += 1
        if instance[Keys.is_alive]:
            conf_stat['a_count'] += 1
        if instance[Keys.is_prepared]:
            conf_stat['p_count'] += 1

    # decide if configurations are prepared and active
    for conf_data in configurations.itervalues():
        if conf_data['a_count'] > 0:
            conf_data[Keys.is_alive] = 1  # configuration is active if there is one active instance
        if conf_data['p_count'] == conf_data['i_count']:
            conf_data[Keys.is_prepared] = 1  # configuration is prepared if all of its instances are prepared

    # print("=== instances ===")
    # pprint(instances)
    # print("=== configurations ===")
    # pprint(configurations)

    return configurations, instances


def ensure_gencfg_tags(tags, properties):
    tags_set = set(tags.split(' '))
    gencfg_group = properties.get('GENCFG_GROUP')
    gencfg_version = properties.get('GENCFG_RELEASE')
    gencfg_revision = properties.get('GENCFG_REVISION')

    if gencfg_version == 'trunk':
        if gencfg_revision:
            gencfg_version = 'trunk-' + gencfg_revision
        else:
            gencfg_version = None

    if 'a_topology_group-' not in tags and gencfg_group:
        tags_set.add('a_topology_group-' + gencfg_group)
        tags_set.add(gencfg_group)
    if 'a_topology_version-' not in tags and gencfg_version:
        tags_set.add('a_topology_version-' + gencfg_version)
        tags_set.add('a_topology_' + gencfg_version)

    return ' '.join(tags_set)


def get_interfaces():
    uname = os.uname()[0]

    out, _ = subprocess.Popen(["ifconfig -a"], shell=True, stdout=subprocess.PIPE).communicate()

    interfaces = {}
    for line in out.split('\n'):
        if line.strip() == '':
            continue

        if not line.startswith(' ') and not line.startswith('\t'):
            dev = line.split(' ')[0].split(':')[0]

            if dev != '':
                interfaces[dev] = {}

                if uname == 'Linux':
                    interfaces[dev]['mac'] = line.strip().split(' ')[-1]
                elif uname == 'FreeBSD':
                    interfaces[dev]['mtu'] = int(line.strip().split(' ')[-1])
        else:
            line = line.strip()

            if line.startswith('inet6'):
                try:
                    v6 = line.replace('addr:', '').replace('  ', ' ').split(' ')[1].replace('/64', '')
                except KeyError:
                    continue

                if v6.startswith('fe80::'):
                    continue

                interfaces[dev][6] = v6

            elif line.startswith('inet'):
                try:
                    v4 = line.replace('addr:', '').replace('  ', ' ').split(' ')[1]
                except KeyError:
                    continue

                interfaces[dev][4] = v4

            elif line.startswith('ether') and uname == 'FreeBSD':
                interfaces[dev]['mac'] = line.strip().split(' ')[-1]

            elif line != line.split('MTU:')[0]:
                interfaces[dev]['mtu'] = int(line.split('MTU:')[1].split(' ')[0])

    for dev in ['lo', 'lo0', 'lo1', 'ipfw0']:
        interfaces.pop(dev, None)

    return interfaces


def get_cgroup_porto(port, ripath):
    pidfiles = []
    for dirpath, _, files in os.walk(str(ripath + '/pids')):
        pidfiles = [os.path.join(dirpath, f) for f in files]
        break

    for pidfile in pidfiles:
        # noinspection PyBroadException
        try:
            with open(pidfile) as f:
                pid = int(f.read())
        except Exception:
            continue

        out, _ = subprocess.Popen(
            ["ps -p {0} ww | grep {1} 2>&1 > /dev/null".format(pid, port) +
             "&& cat /proc/{0}/cgroup ".format(pid) +
             "| awk -F '/' '{print $NF}' | sort | uniq"],
            shell=True,
            stdout=subprocess.PIPE
        ).communicate()

        out = out.strip()
        if out:
            if out.split('@')[0] != out:
                return 'slot' + out.split('@')[0]
            return out

    return ''


def collect_bsconfig_data(configurations, instances):
    # configurations
    try:
        if CONFIG_INSTALL_PATH.check(exists=0) or CONFIG_INSTALL_PATH.check(dir=0):
            return
    except py.error.Error:
        return

    # load active configurations' names
    active_conf_names = set()
    if CONFIG_ACTIVE_PATH.check(exists=1) and CONFIG_ACTIVE_PATH.check(dir=1):
        for pp in CONFIG_ACTIVE_PATH.listdir():
            rpp = pp.realpath()
            bpp = rpp.basename
            active_conf_names.add(bpp)

    cseen = set()
    for path in CONFIG_INSTALL_PATH.listdir():
        try:
            rpath = path.realpath()
            conf = rpath.basename
            if conf in cseen:
                continue
            cseen.add(conf)
            if not rpath.check(exists=1):
                continue
            if not rpath.check(dir=1):
                continue
            configurations[conf] = {Keys.is_alive: 0, Keys.is_prepared: 0, 'is_iss': 0}
            statefile = rpath.join('configuration.state')
            if not statefile.check(exists=1):
                continue
            for line in statefile.readlines():
                line = line.strip()
                if not line:
                    continue
                if ':' in line:
                    key, value = line.split(':', 1)
                    key, value = key.strip(), value.strip()
                    if key == 'prepare' and value == 'OK':
                        configurations[conf][Keys.is_prepared] = 1
                        break
            if conf in active_conf_names:
                configurations[conf][Keys.is_alive] = 1
        except py.error.Error:
            continue

        # prepare instances map with base fields
        for ipath in rpath.listdir():
            try:
                ripath = ipath.realpath()
                name = ripath.basename
                if ':' not in name:
                    continue
                host, port = name.split(':', 1)
                instance_dir = str(ripath)

                instance = {Keys.is_alive: 0, Keys.models: {},
                            Keys.multiplier: 0.0, Keys.is_prepared: 0,
                            Keys.shard_name: '', Keys.configurations: '',
                            'host': host, 'port': port, 'name': name,
                            'instance_dir': instance_dir,
                            'conf': conf, }
                instances.setdefault(name, instance)

                if conf in active_conf_names:
                    instances[name]['instance_dir_a'] = instance_dir
            except py.error.Error:
                continue

    process_bsconfig_instance_state_file(instances)
    collect_bsconfig_itypes(instances)


def instance_state():
    configurations = {}
    instances = {}
    result = {Keys.report_version: REPORT_VERSION, Keys.bsconfig_sync_on: 1, Keys.configurations: configurations,
              Keys.instances: instances, 'iss-agent-ok': True, 'net': get_interfaces()}

    # interfaces

    # bsconfig_sync
    if DB_BSCONFIG_STATE.check(exists=1) and DB_BSCONFIG_STATE.check(dir=1):
        for path in DB_BSCONFIG_STATE.listdir():
            filename = path.realpath().basename
            if 'syncdisable.txt' in filename or 'syncshardsdisable.txt' in filename:
                result[Keys.bsconfig_sync_on] = 0

    # coredumps
    try:
        for path in py.path.local('/coredumps').listdir():
            rpath = path.realpath()
            result['cd'][rpath.basename] = os.stat(str(rpath)).st_mtime
    except:
        pass

    collect_bsconfig_data(configurations, instances)

    collect_iss_instances_info(result, configurations, instances)

    collect_instance_info_from_running_search(instances)

    collect_instance_cgroups(instances)

    for s in instances.itervalues():
        if s[Keys.is_alive] == 1:
            if 'conf' in s and s['conf'] in configurations and configurations[s['conf']]['is_iss'] == 1:
                configurations[s['conf']][Keys.is_alive] = 1

    i = {}
    for k in instances.keys():
        v = {}
        for f in INSTANCE_FIELDS:
            if f in instances[k]:
                v[f] = instances[k][f]
        i[k.split('.')[0].split(':')[0] + ':' + k.split(':')[-1]] = v
    result[Keys.instances] = i

    c = {}
    for k in configurations.keys():
        v = {}
        for f in [Keys.is_alive, Keys.is_prepared, 'is_iss']:
            v[f] = configurations[k].get(f, 0)
        c[k] = v
    result[Keys.configurations] = c

    return result


def collect_iss_instances_info(result, configurations, instances):
    try:
        # print("collect_iss_instances_info:")
        # pprint(instances)
        iss_configurations, iss_instance_dict = get_iss_data()
        configurations.update(iss_configurations)

        # TODO: check logic for transforming instance list to dictionary with instance losing
        for host_port, iss_instance in iss_instance_dict.iteritems():
            if host_port in instances and instances[host_port][Keys.is_alive] == 1:
                continue  # do not overwrite active instance

            instances[host_port] = iss_instance
    except AgentException:
        result['iss-agent-ok'] = False
    except Exception:
        pass


def collect_bsconfig_itypes(instances):
    # noinspection PyBroadException
    try:
        p = subprocess.Popen(['bsconfig', 'listtags', '--yasm-format'], stdout=subprocess.PIPE)
        for line in p.stdout:
            host_port = line.split('@')[0]
            instances[host_port][Keys.all_tags] = line.split('\t')[1]
            if host_port in instances:
                match = re.search(r'a_itype_([a-z]*)', line)
                if match:
                    instances[host_port][Keys.itype] = match.group(1)
    except:
        pass


def process_bsconfig_instance_state_file(instances):
    for instance in instances.itervalues():
        try:
            ripath = py.path.local(instance['instance_dir'])
            istatefile = ripath.join('instance.state')
            if not istatefile.check(exists=1):
                continue
            config_ok = False
            resources_ok = False
            for line in istatefile.readlines():
                if ':' in line:
                    key, value = line.split(':', 1)
                    key, value = key.strip(), value.strip()
                    if key == 'config' and value == 'OK':
                        config_ok = True
                    if key == 'resources' and value == 'OK':
                        resources_ok = True
                    if config_ok and resources_ok:
                        instance[Keys.is_prepared] = 1
                        break
        except py.error.Error:
            continue

    return instances


def collect_instance_info_from_running_search(instances):
    for instance in instances.itervalues():
        host = instance.get('mtn_ip') or instance['host']
        port = instance['port']
        instance_dir = instance['instance_dir']
        itype = instance.get(Keys.itype, None)
        try:
            if itype == 'upper' or itype == 'newsupper':
                instance.update(upper_version_info(host, port, url='/v?json=1'))
            elif itype == 'balancer':
                instance.update(get_common_version_info('127.0.0.1', port, url='/admin?action=version'))
                if 'uaas' in instance['tags']:
                    instance.update(get_balancer_experiments('127.0.0.1', port, url='/admin/events/xmlwrapcall/report'))
            elif itype == 'noapache':
                instance.update(get_common_version_info(host, port, url='/yandsearch?info=getversion'))
                instance.update(rearrange_version_info(host, port, url='/yandsearch?info=getrearrangeversion'))
                instance.update(sandox_task_info(host, port, url='/yandsearch?info=sandboxtaskid'))
            elif itype == 'dataupdaterd':
                instance.update(get_data_updater_version_info(host, port, url='/v'))
            elif itype == 'shardtool':
                shardtool_data = {
                    'shard_size': get_shardtool_shard_size(instance_dir),
                    'shard_status': get_shardtool_shard_status('127.0.0.1', port),
                }
                instance.setdefault(Keys.other_resources, {})
                instance[Keys.other_resources].update(shardtool_data)
            elif itype in YANDSEARCH_AWARE_ITYPES:
                instance.update(get_common_version_info(host, port, url='/yandsearch?info=getversion'))
                instance.update(sandox_task_info(host, port, url='/yandsearch?info=sandboxtaskid'))

            if instance[Keys.is_alive] == 1:
                if itype == 'noapache':
                    if not instance.has_key(Keys.other_resources):
                        instance[Keys.other_resources] = {}
                    instance[Keys.other_resources].update(noapache_config_info(host, int(port)))
                if itype == 'base':
                    # '/yandsearch?info=getconfig'
                    instance.update(base_config_info(host, int(port)))
                    # '/yandsearch?info=rankingmodels'
                    # XXX: models disabled due to their innumerous size!
                    # instance.update(ranking_info(host, int(port)))
                    # '/supermind?action=getmult'
                    instance.update(supermind_info(host, int(port)))
                    # '/yandsearch?info=checkconfig:base-frm-md5'
                    instance.update(models_info(host, int(port), instance_dir))
        except Exception:
            pass

    return instances


def collect_instance_cgroups(instances):
    iseen = {}

    for instance in instances.itervalues():
        name = instance['name']

        try:
            cgroup = get_cgroup_porto(instance['port'], instance.get('instance_dir_a', instance['instance_dir']))
        except Exception:
            cgroup = ''

        c = name in iseen

        if len(cgroup):
            iseen[name] = cgroup
        elif not c:
            iseen[name] = ''

        if c:
            continue

    for instance_name, cgroup in iseen.iteritems():
        instances[instance_name][Keys.configurations] = cgroup

    return instances


def get_info_by_http(host, port, tries=3, url='/yandsearch?info=getversion', timeout=5):
    net_families = [socket.AF_INET, socket.AF_INET6]
    for net_family in net_families:
        try:
            # if this succeeds, we have ip rather than host
            socket.inet_pton(net_family, host)
        except socket.error:
            pass
        else:
            net_families = [net_family]
            break

    for net_family in net_families:
        for try_ in range(1, tries + 1):
            try:
                sock = socket.socket(net_family, socket.SOCK_STREAM)
                sock.settimeout(timeout)
                sock.connect((host, int(port)))
                sock.send('GET %s HTTP/1.0\r\n' % (url,))
                sock.send('Host: %s\r\n' % (host,))
                sock.send('User-Agent: %s\r\n' % (USER_AGENT,))
                sock.send('\r\n')
                buf = []
                while 1:
                    data = sock.recv(8192)
                    if not data:
                        break
                    buf.append(data)
                data = ''.join(buf)

                head = data.split('\r\n\r\n', 1)[0].split('\r\n')[0]
                if 'HTTP/1.1 200' in head or 'HTTP/1.0 200' in head:
                    return data.split('\r\n\r\n', 1)[1]
                else:
                    return 'NonHTTP200'
            except socket.error:
                time.sleep(0.1)
    return ''


def parse_svn_info(info):
    data = {}

    def update(line, prefix, key):
        if prefix in line:
            data[key] = line.strip().split()[-1]

    for line in info.split('\n'):
        update(line, 'URL: ', Keys.svn_url)
        update(line, 'Last Changed Rev: ', Keys.revision)
        update(line, 'Revision: ', Keys.revision)
        update(line, 'Binary\'s MD5', Keys.md5)
        update(line, 'Sandbox task:', Keys.sandbox_task)
    return data


def get_common_version_info(host, port, url):
    info = get_info_by_http(host, port, tries=3, url=url)
    data = {}
    if info.__len__() > 0:
        data[Keys.is_alive] = 1
        data.update(parse_svn_info(info))
    return data


def get_data_updater_version_info(host, port, url):
    info = get_info_by_http(host, port, tries=3, url=url)
    data = {}
    if info.__len__() > 0:
        data[Keys.is_alive] = 1
        data[Keys.svn_url] = info
    return data


def get_balancer_experiments(host, port, url):
    info = get_info_by_http(host, port, tries=3, url=url)
    data = {Keys.other_resources: {'balancer_exp': {}}}
    # noinspection PyBroadException
    try:
        for l in parseString(info).toprettyxml().split('\n'):
            if '<exp_config_version>' in l:
                exp = l.split('<exp_config_version>')[1].split('</exp_config_version>')[0]
                data[Keys.other_resources]['balancer_exp'].setdefault(exp, 0)
                data[Keys.other_resources]['balancer_exp'][exp] += 1
    except:
        pass
    return data


def extract_upper_data(data, key):
    if isinstance(data, dict) and data.has_key(key):
        if isinstance(data[key], unicode):
            return data[key].encode()
    return None


def upper_version_info(host, port, url):
    info = get_info_by_http(host, port, tries=5, url=url, timeout=10)
    data = {}
    if info.__len__() > 0:
        data[Keys.is_alive] = 1
        data[Keys.other_resources] = {}
        upper_data = json.loads(info)
        for resource in ['report', 'apache', 'upper', 'templates']:
            resource_info = {}
            if upper_data.has_key(resource):
                resource_info[Keys.revision] = extract_upper_data(upper_data[resource], 'revision')
                resource_info[Keys.sandbox_task] = extract_upper_data(upper_data[resource], 'sandbox_task')
                resource_info[Keys.svn_url] = extract_upper_data(upper_data[resource], 'version')
            data[Keys.other_resources][resource] = resource_info
    return data


def get_base_url(host, port):
    if ':' in host:
        host = '[{}]'.format(host)
    return "http://{host}:{port}".format(host=host, port=port)


def rearrange_version_info(host, port, url):
    info = get_info_by_http(host, port, tries=3, url=url)
    rearrange_data = {}
    if info.__len__() > 0:
        rearrange_data['rearrange'] = parse_svn_info(info.split('RearrangeDynamicData')[0])
        rearrange_data['rearrange_dynamic'] = parse_svn_info(info.split('RearrangeDynamicData')[1])
    data = {Keys.other_resources: rearrange_data}
    return data


def sandox_task_info(host, port, url):
    info = get_info_by_http(host, port, tries=3, url=url)
    data = {}
    if info.__len__() > 0:
        data[Keys.sandbox_task] = info.strip()[:30]
    return data


def base_config_info(host, port):
    info = get_info_by_http(host, port, tries=3, url='/yandsearch?info=getconfig')
    data = {}
    for a in info.split('\n'):
        if 'IndexDir ' in a:
            try:
                data[Keys.shard_name] = \
                    a.strip(' \t\n\r').split(' ')[1].replace('/db/BASE/', '').replace('/ssd/', '').split('/')[0]
            except IndexError:
                data[Keys.shard_name] = ''
            break
    return data


def noapache_config_info(host, port):
    info = get_info_by_http(host, port, tries=3, url='/yandsearch?info=getconfig')
    data = {'config': parse_svn_info(info)}
    return data


def models_info(host, port, conf_dir):
    if 'mmeta' in conf_dir:
        info = get_info_by_http(host, port, tries=3, url='/yandsearch?info=checkconfig:mmiddle-frm-md5;deep:0')
    else:
        info = get_info_by_http(host, port, tries=3, url='/yandsearch?info=checkconfig:base-frm-md5')
    data = {}
    if 'error' not in info:
        data[Keys.models_md5] = info[(info.find('<pvalue>') + 8):info.find('</pvalue>')]
    return data


def ranking_info(host, port):
    info = get_info_by_http(host, port, tries=3, url='/yandsearch?info=rankingmodels')
    data = {}
    models = {}
    try:
        document = parseString(info)
        for i in document.getElementsByTagName("i2"):
            for k, v in i.attributes.items():
                if str(k) == 'c1':
                    c1 = str(v)
                if str(k) == 'c2':
                    c2 = str(v)
            if c1.count(':') == 2:
                models.setdefault(c1, c2)
        data[Keys.models] = models
    except Exception:
        data[Keys.models] = {}
    return data


def supermind_info(host, port):
    info = get_info_by_http(host, port, tries=3, url='/supermind?action=getmult')
    data = {}
    try:
        a = float(info)
        data[Keys.multiplier] = a
    except ValueError:
        data[Keys.multiplier] = 0.0
    return data


def get_shardtool_shard_size(instance_dir):
    p = subprocess.Popen(
        'du -s --exclude ".tmp" {}/primus-*'.format(instance_dir),
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        shell=True,
    )
    out, err = p.communicate()

    if p.returncode:
        return 'Failed to determine shard size in [{}]: {}'.format(instance_dir, err)

    return out.split()[0]


def get_shardtool_shard_status(host, port):
    return get_info_by_http(host, port, tries=3, url='/status_message')


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    return parser.parse_args()


def main():
    args = parse_args()
    result = instance_state()
    if args.format == 'pretty':
        pprint(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
