from __future__ import absolute_import, print_function, division

import argparse
import json
import msgpack
import os
import py
import socket
import time

from pprint import pprint
import subprocess


class Keys:
    is_alive = 'a'  # active configuration, alive instance
    configurations = 'c'  # list of all configurations
    instances = 'i'  # list of all instances
    models = 'm'  # ranking models of base search instance
    md5 = 'md5'  # instance binary's md5
    multiplier = 'mu'  # base search instance multiplier (ruchka) - /supermind?action=getmult
    models_md5 = 'mv'  # ranking models MD5 of base search instance
    is_prepared = 'p'  # prepared configuration, prepared instance
    revision = 'r'  # instance binary's svn revision
    itype = 't'  # instance type
    all_tags = 'tags'  # all instance tags
    report_version = 'v'  # report version


REPORT_VERSION = 4
CONFIG_INSTALL_PATH = py.path.local('/db/bsconfig/configinstall')
CONFIG_ACTIVE_PATH = py.path.local('/db/bsconfig/configactive')
DB_BSCONFIG_STATE = py.path.local('/db/bsconfig/state')
ISS_AGENT_CONFIGURATIONS_PATH = '/db/iss3/configurations/'
ISS_AGENT_INFO_PORT = 25536
ISS_AGENT_HTTP_TIMEOUT = 20

USER_AGENT = "Heartbeat/Skynet"


class AgentException(StandardError):
    pass


def load_iss_instance_states(url):
    def _decode_list(data):
        rv = []
        for item in data:
            if isinstance(item, unicode):
                item = item.encode()
            elif isinstance(item, list):
                item = _decode_list(item)
            elif isinstance(item, dict):
                item = _decode_dict(item)
            rv.append(item)
        return rv

    def _decode_dict(data):
        rv = {}
        for key, value in data.iteritems():
            if isinstance(key, unicode):
                key = key.encode()
            if isinstance(value, unicode):
                value = value.encode()
            elif isinstance(value, list):
                value = _decode_list(value)
            elif isinstance(value, dict):
                value = _decode_dict(value)
            rv[key] = value
        return rv

    json_data = get_info_by_http(socket.getfqdn(), ISS_AGENT_INFO_PORT, url=url, timeout=ISS_AGENT_HTTP_TIMEOUT)
    try:
        return json.loads(json_data, object_hook=_decode_dict)
    except (ValueError, IndexError):
        raise AgentException()

def run_command(cmds):
    popen = subprocess.Popen(
        cmds,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
    )
    out, _ = popen.communicate()

    return out.strip(), popen.returncode


def portoctl_point(container, datatype):
    t = time.time()
    out, _ = run_command(['portoctl', 'dget', container, datatype])
    try:
        v = int(out)
    except ValueError:
        v = -1

    return t, v


def portoctl_measure(container, datatype, dt=10):
    p0 = portoctl_point(container, datatype)
    time.sleep(dt)
    p1 = portoctl_point(container, datatype)

    return {'p0': p0, 'p1': p1, 'dt': dt}


def get_porto_counters(instance):
    if instance['type'] == 'bsconfig':
        porto_container = '%s:%s' % (instance['conf'], instance['port'])
    elif instance['type'] == 'iss':
        porto_container = instance['containerPath']
    else:
        porto_container = None

    if not porto_container:
        return

    return {
        'mem': portoctl_point(porto_container, 'memory_usage')
    }


def get_iss_data():
    all_instances = load_iss_instance_states('/instances')

    instances = {}

    for data in all_instances:
        port = data['slot'].split('@', 1)[0]
        instances[port] = {
            'conf': data['configurationId'],
            'containerPath': data['containerPath'].split('/')[0],
            'tags': data['instanceData']['properties/tags'],
            'type': 'iss',
        }

    return instances


def get_bsconfig_data():
    instances = {}

    # configurations
    try:
        if CONFIG_INSTALL_PATH.check(exists=0) or CONFIG_INSTALL_PATH.check(dir=0):
            return
    except py.error.Error:
        return

    cseen = set()
    for path in CONFIG_INSTALL_PATH.listdir():
        try:
            rpath = path.realpath()
            conf = rpath.basename
            if conf in cseen:
                continue
            cseen.add(conf)
            if not rpath.check(exists=1):
                continue
            if not rpath.check(dir=1):
                continue
        except py.error.Error:
            continue

        # prepare instances map with base fields
        for ipath in rpath.listdir():
            try:
                ripath = ipath.realpath()
                name = ripath.basename
                if ':' not in name:
                    continue
                host, port = name.split(':', 1)

                instances[port] = {
                    'conf': conf,
                    'type': 'bsconfig',
                    'port': port,
                }

            except py.error.Error:
                continue

    collect_bsconfig_itypes(instances)
    return instances


def instance_state():
    instances = get_iss_data()
    instances.update(get_bsconfig_data())
    for instance in instances.itervalues():
        instance['counters'] = get_porto_counters(instance)

    result = {Keys.report_version: REPORT_VERSION, Keys.instances: instances}

    return result


def collect_bsconfig_itypes(instances):
    try:
        p = subprocess.Popen(['bsconfig', 'listtags', '--yasm-format'], stdout=subprocess.PIPE)
        for line in p.stdout:
            port = line.split('@')[0].split(':')[1]
            instances[port][Keys.all_tags] = line.split('\t')[1]
    except:
        pass


def get_info_by_http(host, port, tries=3, url='/yandsearch?info=getversion', timeout=5):
    for net_family in [socket.AF_INET, socket.AF_INET6]:
        for try_ in range(1, tries + 1):
            try:
                sock = socket.socket(net_family, socket.SOCK_STREAM)
                sock.settimeout(timeout)
                sock.connect((host, int(port)))
                sock.send('GET %s HTTP/1.0\r\n' % (url,))
                sock.send('Host: %s\r\n' % (host,))
                sock.send('User-Agent: %s\r\n' % (USER_AGENT,))
                sock.send('\r\n')
                buf = []
                while 1:
                    data = sock.recv(8192)
                    if not data:
                        break
                    buf.append(data)
                data = ''.join(buf)
                if data.split('\r\n\r\n', 1)[0].split('\r\n')[0] == 'HTTP/1.1 200 OK':
                    return data.split('\r\n\r\n', 1)[1]
                else:
                    return 'NonHTTP200'
            except socket.error:
                time.sleep(0.1)
    return ''


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    return parser.parse_args()


def main():
    args = parse_args()
    result = instance_state()
    if args.format == 'pretty':
        pprint(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
