#!/skynet/python/bin/python
import argparse
import msgpack
import requests
import json
import os
import stat

from os.path import exists
from pprint import pprint


ISS_VERSION_URL = 'http://localhost:25536/version'
ISS_STATUS_URL = 'http://localhost:25536/status'
ISS_INSTANCES_URL = 'http://localhost:25536/instances'
ISS_CONFIG_FILE = '/db/iss3/application.conf'
PORTO_OPTIONS_FILE = '/db/iss3/porto.options'
VM_OPTIONS_FILE = '/db/iss3/iss-agent.vmoptions'


def _do_request(url, timeout):
    exc = None
    try:
        r = requests.get(url, timeout=timeout, verify=False)
        if r.status_code != requests.codes.ok:
            return {'error': str(r.status_code)}
        return r.json()
    except requests.exceptions.Timeout:
        exc = 'timeout'
    except requests.exceptions.ConnectionError as e:
        exc = str(e)
    finally:
        if exc == 'timeout':
            return {'error': 'timeout'}

        if exc is not None:
            return {'error': exc}


def get_version():
    response = _do_request(ISS_VERSION_URL, timeout=10)
    return response.get('version', 'N/A')


def get_application_conf():
    with open(ISS_CONFIG_FILE) as f:
        conf = json.load(f)
        return conf


def get_status():
    return _do_request(ISS_STATUS_URL, timeout=10)


def get_porto_options():
    options = {}
    with open(PORTO_OPTIONS_FILE) as f:
        for l in f:
            if l.startswith('#'):
                # ignore comments
                continue
            chunks = l.split('=')
            if len(chunks) > 1:
                options[chunks[0]] = '='.join(chunks[1:]).strip()
    return options


def get_vm_options():
    options = []
    with open(VM_OPTIONS_FILE) as f:
        found_overrides = False
        for l in f:
            if l.startswith('#-- start managed zone --'):
                found_overrides = True
            elif l.startswith('#-- end managed zone --'):
                found_overrides = False
            else:
                if found_overrides and l.startswith('-'):
                        options.append(l.strip('-\n\t '))
    return options


def get_iss_state_size():
    return os.stat("/iss3/state3").st_size if os.path.exists('/iss3/state3') else 0


def get_missing_dirs():
    if stat.S_IMODE(os.stat("/db/iss3/instances").st_mode) == 0o700:
        return ['Failed to access instance dirs',]

    result = _do_request(ISS_INSTANCES_URL, timeout=10)
    if 'error' in result:
        raise Exception('ERROR: {}'.format(result['error']))

    dirs = list()
    for r in result:
        if r.get('instanceData', {}).get('properties/DEPLOY_ENGINE', '') == 'YP_LITE':
            continue

        instance_dir  = r.get('instanceDir', "")
        if instance_dir:
            if not os.path.isdir(instance_dir) or not os.path.exists(instance_dir):
                dirs.append(instance_dir)

    return {
        'count': len(dirs),
        'dirs': dirs
    }


def get_misc():
    return {'state_size': get_iss_state_size(),
            'missing_dirs': get_missing_dirs()}


def get_iss_info():
    return {
        'version': get_version(),
        'config': get_application_conf(),
        'porto_options': get_porto_options(),
        'vm_options': get_vm_options(),
        'status': get_status(),
        'misc': get_misc()
    } if exists(ISS_CONFIG_FILE) else {
        'version': 'Not installed',
        'config': {},
        'porto_options': {},
        'vm_options': [],
        'status': {},
        'misc': {}
    }


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    return parser.parse_args()


def main():
    args = parse_args()
    try:
        result = get_iss_info()
        # reset old errors
        result['error'] = ''
    except BaseException as ex:
        result = {'error': str(ex)}

    if args.format == 'pretty':
        pprint(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
