from __future__ import absolute_import, print_function, division

import argparse
import msgpack
import py
import datetime
import shlex
import time
import subprocess
from threading import Timer
from collections import namedtuple

from pprint import pprint


REPORT_VERSION = 1

CommandResult = namedtuple('CommandResult', ['returncode', 'out', 'err', 'has_timeout', 'elapsed'])


def kill_process(process, dto):
    """
    timed out recipe from
      https://stackoverflow.com/questions/1191374/using-module-subprocess-with-timeout/10768774#10768774
    """
    dto["value"] = True
    process.kill()


def run_command(args, lines=False, timeout_sec=60, exception_on_timeout=True):
    if type(args) == str:
        args = shlex.split(args)

    cmdline = " ".join(args)

    started_time = time.time()

    proc = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

    timeout_dto = {"value": False}
    timer = Timer(timeout_sec, kill_process, [proc, timeout_dto])

    timer.start()
    out, err = proc.communicate()
    timer.cancel()

    if lines:
        out = filter(None, out.splitlines())
        err = filter(None, err.splitlines())

    elapsed_time = time.time() - started_time

    if exception_on_timeout and timeout_dto["value"]:
        raise Exception("got timeout (%r sec) on [%s]" % (timeout_sec, cmdline))

    return CommandResult(returncode=proc.returncode, out=out, err=err, has_timeout=timeout_dto["value"],
                         elapsed=elapsed_time)


def parse_dt_str(dt_str):
    """

    Wed Dec  1 17:33:07 2021
    """
    # TODO: date string parsing must be changed !!!

    ts = int(run_command('date -d"{}" +%s'.format(dt_str)).out)
    return datetime.datetime.fromtimestamp(ts)


def get_user_sessions():
    """
    Example command output:
     ~ who
     i-dyachkov pts/0        2021-12-22 14:21 (2a02:6b8:81:1:2cf0:11d0:13ac:f93e)
     i-dyachkov pts/9        2021-12-22 16:27 (2a02:6b8:81:1:2cf0:11d0:13ac:f93e)
    """
    user_sessions = {}
    user_sessions_length = {}
    wtmp_begin_dt = None
    wtmp_files = ['wtmp.1', 'wtmp']
    command_last = "last --time-format iso --fullnames --nohostname"

    for wtmp_file in wtmp_files:
        result = run_command(command_last + " -f /var/log/{}".format(wtmp_file), lines=True)
        if result.err:
            continue

        for s in result.out:
            line = s.strip()
            if line.startswith('wtmp'):
                if wtmp_begin_dt:
                    continue
                wtmp_begin = line.replace('{} begins '.format(wtmp_file), '').strip()
                wtmp_begin_dt = parse_dt_str(wtmp_begin)
                continue
            parts = [p for p in line.split(' ') if p.strip() and p.strip() != '-']
            login = parts[0]
            if login in ('reboot', 'shutdown', 'runlevel'):
                continue
            is_still_logged_in = 'still logged in' in line
            is_session_crash = ' - crash' in line
            is_gone_no_logout = 'gone - no logout' in line
            user_session_start = parse_dt_str(parts[2])

            if is_still_logged_in or is_session_crash or is_gone_no_logout:
                user_session_end = datetime.datetime.now()
            else:
                try:
                    user_session_end = parse_dt_str(parts[3])
                except Exception:
                    user_session_end = datetime.datetime.now()

            if login not in user_sessions or user_sessions[login] < user_session_end:
                user_sessions[login] = user_session_end

            if login not in user_sessions_length:
                user_sessions_length[login] = datetime.timedelta()

            user_sessions_length[login] += (user_session_end - user_session_start)

    return {
        'user_sessions': dict((u, d.isoformat() + "Z") for u, d in user_sessions.items()),
        'wtmp_begin_dt': wtmp_begin_dt.isoformat() + "Z",
        'user_sessions_seconds_length': dict((u, int(d.total_seconds())) for u, d in user_sessions_length.items()),
    }


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    return parser.parse_args()


def main():
    args = parse_args()

    result = {
        'report_version': REPORT_VERSION,
    }
    result.update(**get_user_sessions())
    if args.format == 'pretty':
        pprint(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
