#!/skynet/python/bin/python

from __future__ import absolute_import, print_function, division

import argparse
import os

from api.srvmngr import getRoot as get_root


def handle_metrics(implementation, rundir):
    root = get_root()

    log_filename = os.path.join(root, 'var', 'log', '{}-server.log'.format(implementation))
    if not os.path.exists(log_filename):
        return None

    state_filename = os.path.join(rundir, '{}-restarts'.format(implementation))
    with open(state_filename, 'a+') as stat_file:
        try:
            last_report_position = int(stat_file.readline())
            last_report_time = stat_file.readline()
        except ValueError:
            last_report_position = -1
            last_report_time = ''
            pass

    results = []

    mark = (
        'Initializing cqudp' if implementation == 'cqudp'
        else '======================= Started as ROOT'
    )

    with open(log_filename, 'r') as log:
        if last_report_position >= 0:
            log.seek(last_report_position)
            line = log.readline()

            # check whether log has been rotated or removed
            if not line.startswith(last_report_time):
                # start from the first line in the file
                log.seek(0)

        # Read and parse data from the file from the specified position
        # Example: 2014-09-05 16:15:09.724 [INFO] [cqueue:ya.skynet.cqueue]  MSG
        # log line consists of 5 parts at least
        for line in log:
            parts = line.split(' ', 4)

            if len(parts) < 5:
                # invalid format we cannot parse this line
                continue

            if parts[4].strip().startswith(mark):
                results.append((parts[0], parts[1]))

        # write last sent report into local file
        if len(results) > 0:
            with open(state_filename, 'w') as stat_file:
                # write beginning of the last processed line
                pos = log.tell() - len(line)
                stat_file.write(str(pos) + '\n')
                stat_file.write(parts[0] + ' ' + parts[1])

    return results


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    parser.add_argument('-i', '--implementation', choices=('cqueue', 'cqudp'))
    parser.add_argument(
        '--rundir', default=None, type=str, help='heartbeat-client rundir',
    )

    args = parser.parse_args()
    if args.rundir is None:
        args.rundir = os.path.join(get_root(), 'var', 'heartbeat-client')

    return args


def main():
    args = parse_args()

    info = handle_metrics(args.implementation, args.rundir)
    if not info:
        # there is nothing to send
        return

    if args.format == 'msgpack':
        import msgpack
        print(msgpack.dumps(info))
    elif args.format == 'pretty':
        from pprint import pprint as pp
        pp(info)


if __name__ == '__main__':
    main()
