#!/skynet/python/bin/python

from __future__ import absolute_import, print_function, division

import argparse
import msgpack
import os

from api.srvmngr import getRoot as get_root


def handle_overhead(parts):
    # Format: <type> <time> total_hosts: <value> total_time: <value> total_user_time: <value> number_of_results: <value>
    #         delta: <value>
    if len(parts) != 12:
        # ignore invalid format
        return None

    try:
        return {'type': parts[0],
                'time': float(parts[1]),
                'total_hosts': int(parts[3]),
                'total_results': int(parts[9]),
                'delta': float(parts[11])}

    except ValueError:
        # failed to convert one of parts -> wrong format, ignore it
        return None


def handle_error(parts):
    # Format: <type> <time> <total_hosts> <number of errors> error description
    if len(parts) < 4:
        # ignore invalid format
        return None

    try:
        return {'type': parts[0],
                'time': float(parts[1]),
                'total_hosts': int(parts[2]),
                'err_count': int(parts[3])}

    except ValueError:
        # failed to convert one of parts -> wrong format, ignore it
        return None


def handle_restypes(parts):
    # Format: <<TypeName>:<count>>...
    data = []
    if len(parts) < 2:
        return

    for part in parts[1:]:
        try:
            name, count = part.split(':')
            count = int(count)
            data.append({
                'name': name,
                'count': count,
            })
        except (ValueError, TypeError):
            pass

    return {
        'type': 'ResTypes',
        'records': data,
    }


def handle_unknown(parts):
    # do nothing for unknown metric
    return None


def handle_metrics(implementation, mode, rundir):
    # Find {root}/var/log file
    info_filename = '{}/var/log/{}-metrics.log'.format(get_root(), implementation)
    if not os.path.exists(info_filename):
        # there is nothing to report
        return None

    # read last sent report from the local file
    stat_filename = '{}/{}-metrics'.format(
        rundir, implementation if mode == "errors" else "{}-{}".format(implementation, mode))
    with open(stat_filename, 'a+') as stat_file:
        try:
            last_report_position = int(stat_file.readline())
            last_report_time = stat_file.readline()
        except ValueError:
            last_report_position = -1
            last_report_time = ''
            pass

    results = list()
    handlers = {
        'Overhead': handle_overhead,
        'AccessError': handle_error,
        'RuntimeError': handle_error,
        'UserError': handle_error,
    } if mode == "errors" else {
        'ResTypes': handle_restypes,
    }

    with open(info_filename, 'r') as info_file:
        if last_report_position >= 0:
            info_file.seek(last_report_position)
            line = info_file.readline()

            # check whether log has been rotated or removed
            if not line.startswith(last_report_time):
                # start from the first line in the file
                info_file.seek(0)

        parts = list()
        line = ''

        # Read and parse data from the file from the specified position
        # Example: 2014-09-05 16:15:09.724 [INFO] [cqudp:metrics] <Metric Type> ... metric specific data
        # Metric line consists of 5 parts at least
        for line in info_file:
            parts = line.split()

            if len(parts) < 5:
                # invalid format we cannot parse this line
                continue

            # metric line can contain task id
            # Example: 2014-09-05 16:15:09.724 [INFO] [cqudp:metrics] [fe23c6a8] <Metric Type> ... metric specific data
            if parts[4].startswith('['):
                parts = parts[5:]
            else:
                parts = parts[4:]

            handler = handlers.get(parts[0], handle_unknown)

            # pass metric type and metric specific data to the handler
            metric = handler(parts)
            if metric is not None:
                results.append(metric)

        # write last sent report into local file
        if len(results) > 0:
            with open(stat_filename, 'w') as stat_file:
                # write beginning of the last processed line
                pos = info_file.tell() - len(line)
                stat_file.write(str(pos) + '\n')
                stat_file.write(parts[0] + ' ' + parts[1])

    return results


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    parser.add_argument('-i', '--implementation', choices=('cqueue', 'cqudp'))
    parser.add_argument('-m', '--mode', choices=('errors', 'restypes'), default='errors')
    parser.add_argument(
        '--rundir', default=None, type=str, help='heartbeat-client rundir',
    )

    args = parser.parse_args()
    if args.rundir is None:
        args.rundir = os.path.join(get_root(), 'var', 'heartbeat-client')

    return args


def main():
    args = parse_args()

    info = handle_metrics(args.implementation, args.mode, args.rundir)
    if not info:
        # there is nothing to send
        return

    if args.format == 'msgpack':
        print(msgpack.dumps(info))
    elif args.format == 'pretty':
        from pprint import pprint as pp
        pp(info)


if __name__ == '__main__':
    main()
