#!/skynet/python/bin/python

from __future__ import absolute_import, print_function, division

import argparse
import msgpack
import os

from api.srvmngr import getRoot as get_root


def handle_metrics(rundir):
    # Find {root}/var/log file
    info_filename = '{}/var/log/hostresolver.log'.format(get_root())
    if not os.path.exists(info_filename):
        # there is nothing to report
        return None

    # read last sent report from the local file
    stat_filename = '{}/hostresolver-metrics'.format(rundir)
    with open(stat_filename, 'a+') as stat_file:
        try:
            last_report_position = int(stat_file.readline())
            last_report_time = stat_file.readline()
        except ValueError:
            last_report_position = -1
            last_report_time = ''
            pass

    results = list()

    with open(info_filename, 'r') as info_file:
        if last_report_position >= 0:
            info_file.seek(last_report_position)
            line = info_file.readline()

            # check whether log has been rotated or removed
            if not line.startswith(last_report_time):
                # start from the first line in the file
                info_file.seek(0)

        parts = list()
        line = ''

        # Read and parse data from the file from the specified position
        # Example: 2017-05-17 13:27:45.718 [INFO] [resolver:resolver.metrics] [username] 1495016865.717759 C@CONF1 H@GROUP2...
        # Metric line consists of 5 parts at least
        for line in info_file:
            parts = line.strip().split()

            if len(parts) < 5:
                # invalid format we cannot parse this line
                continue

            username = parts[4]
            if username[0] != '[' or username[-1] != ']':
                continue
            else:
                username = username[1:-1]

            timestamp = float(parts[5])
            tokens = parts[6:]

            results.append({
                'username': username,
                'timestamp': timestamp,
                'tokens': tokens,
            })

        # write last sent report into local file
        if len(results) > 0:
            with open(stat_filename, 'w') as stat_file:
                # write beginning of the last processed line
                pos = info_file.tell() - len(line)
                stat_file.write(str(pos) + '\n')
                stat_file.write(parts[0] + ' ' + parts[1])

    return results


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    parser.add_argument(
        '--rundir', default=None, type=str, help='heartbeat-client rundir',
    )

    args = parser.parse_args()
    if args.rundir is None:
        args.rundir = os.path.join(get_root(), 'var', 'heartbeat-client')

    return args


def main():
    args = parse_args()

    info = handle_metrics(args.rundir)
    if not info:
        # there is nothing to send
        return

    if args.format == 'msgpack':
        print(msgpack.dumps(info))
    elif args.format == 'pretty':
        from pprint import pprint as pp
        pp(info)


if __name__ == '__main__':
    main()
